# Doriflow Engine - Fluid Simulation for Blender 3D
# Copyright (C) 2024 Doriflow Team
# This software is licensed under the Creative Commons Attribution-NonCommercial 4.0 International License (CC BY-NC 4.0).

# You are free to:
# -Share: Copy and redistribute the material in any medium or format.
# -Adapt: Remix, transform, and build upon the material.

# UNDER THE FOLOWING TERMS:
# -Attribution: You must give appropriate credit, provide a link to the license, and indicate if changes were made.
# -Appropriate credit should include the following:
#   -The original author's name: Doriflow Team
#   -A link to the original source (if applicable).
#   -A link to the full license: https://creativecommons.org/licenses/by-nc/4.0/.
#   -A clear indication of any changes made, such as: "This material has been modified."
# NonCommercial: You may not use the material for commercial purposes.
# Disclaimer:
# -This simulation engine is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. In no event shall the authors be liable for any claim, damages, or other liability arising from the use of this software.

# For more details, refer to the full license text at:
# https://creativecommons.org/licenses/by-nc/4.0/.

#-----------------------------------------------------------------------------------------------------------------------#
import taichi as ti
import numpy as np
from core_steps import CoreSteps
import math

import taichi.math as tm
# import math
Real = ti.f32
Integer = ti.i32
Byte = ti.i8
Vector2 = ti.types.vector(2, Real)
Vector3 = ti.types.vector(3, Real)
Vector4 = ti.types.vector(4, Real)
Vector3i = ti.types.vector(3, Integer)
Vector2i = ti.types.vector(2, Integer)
Matrix3x3 = ti.types.matrix(3, 3, Real)

class FlowProperties(CoreSteps):
    def __init__(self, particle_system):
        super().__init__(particle_system)
        self.exponent = self.ps.cfg.get_domain("exponent")
        self.stiffness = self.ps.cfg.get_domain("stiffness")
        self.surface_tension = self.ps.cfg.get_domain("surface_tension")
        self.dt[None] = self.ps.cfg.get_domain("timeStepSize")
        self.collision_stiffness=0.1
        self.g_taichi = ti.Vector.field(3, dtype=ti.f32, shape=())
        self.g_taichi[None] = ti.Vector(self.ps.cfg.get_domain("gravitation"))
        self.export_fluid_velocity = self.ps.cfg.get_domain("export_fluid_velocity")
        self.liquid_drag_coefficient = self.ps.cfg.get_domain("liquid_drag_coefficient")
        self.normal_threshold = 0.1
        self.normal_sum_and_count = ti.Vector.field(4, dtype=ti.f32, shape=())
        self.normal_sum_and_count[None] = ti.Vector([0.0, 0.0, 0.0, 0.0])
        self.viscosity_model = 0 #for newtonian fluid
        self.dem_stiffness = 10e3 
        self.dem_friction_coefficient = 0.1  
        self.dem_damping_coefficient = 0.001 
        self.dem_drag_coefficient = 0.1
        self.dem_stiffness = self.ps.cfg.get_domain("dem_stiffness") * 1e3
        self.dem_friction_coefficient = self.ps.cfg.get_domain("dem_friction_coefficient")
        self.dem_damping_coefficient = self.ps.cfg.get_domain("dem_damping_coefficient")
        self.dem_drag_coefficient = self.ps.cfg.get_domain("dem_drag_coefficient")


    @ti.func
    def compute_densities_task(self, p_i, p_j, ret: ti.template()):
        x_i = self.ps.x[p_i]
        if self.ps.material[p_j] == self.ps.material_fluid:
            x_j = self.ps.x[p_j]
            ret += self.ps.m_V[p_j] * self.poly6_kernel((x_i - x_j).norm())
        elif self.ps.material[p_j] == self.ps.material_solid :
            x_j = self.ps.x[p_j]
            ret += self.ps.m_V[p_j] * self.poly6_kernel((x_i - x_j).norm())
    @ti.kernel  
    def compute_densities(self):
        for p_i in ti.grouped(self.ps.x):
            if not self.ps.material[p_i] == self.ps.material_fluid:
                continue
            self.ps.density[p_i] = self.ps.m_V[p_i] * self.poly6_kernel(0.0)
            den = 0.0
            self.ps.for_all_neighbors(p_i, self.compute_densities_task, den)
            self.ps.density[p_i] += den
            self.ps.density[p_i] *= self.fluid_density
            if self.ps.density[p_i] < 0.1 *self.fluid_density:
                self.ps.active[p_i] = 0 
    
    @ti.func
    def compute_densities_solid_task(self, p_i, p_j, ret: ti.template()):
        x_i = self.ps.x[p_i]
        if self.ps.material[p_j] == self.ps.material_solid :
            x_j = self.ps.x[p_j]
            ret += self.ps.m_V[p_j] * self.poly6_kernel((x_i - x_j).norm())
                
    @ti.func
    def compute_pressure_forces_task(self, p_i, p_j, ret: ti.template()):
        x_i = self.ps.x[p_i]
        x_j = self.ps.x[p_j]
        dpi = self.ps.pressure[p_i] / (self.ps.density[p_i] ** 2)
        if self.ps.material[p_j] == self.ps.material_fluid:
            density_i = self.ps.density[p_i]
            density_j = self.ps.density[p_j]   
            dpj = self.ps.pressure[p_j] / (density_j * density_j)
            effective_density = 0.5 * (density_i + density_j)
            ret += -effective_density * self.ps.m_V[p_j] * (dpi + dpj) * self.poly6_kernel_derivative(x_i-x_j)
        elif self.ps.material[p_j] == self.ps.material_solid :
            density_j = self.ps.density[p_j]   
            density_i = self.ps.density[p_i]   
            dpj = self.ps.pressure[p_j] / (density_j * density_j)
            f_p = -density_j * self.ps.m_V[p_j] * self.ps.pressure[p_i] / (density_j * density_i) \
                * self.poly6_kernel_derivative(x_i-x_j)  
                
            v_rel = self.ps.v[p_j] - self.ps.v[p_i]
            v_rel_norm = v_rel.norm() + 1e-5 
            v_rel_dir = v_rel / v_rel_norm  
            A = self.ps.m_V[p_j]**(2/3)  
            drag_force = -0.5 * self.liquid_drag_coefficient * self.ps.density[p_i] * A * v_rel_norm**2 * v_rel_dir  
              
            ret += f_p 
            if self.ps.active[p_i] and self.ps.is_dynamic_rigid_body(p_j):
                self.ps.acceleration[p_j] += -f_p * self.fluid_density / self.ps.density[p_j]
                object_j = self.ps.object_id[p_j]
                center_of_mass_j = self.ps.rigid_body_centers_of_mass[object_j]
                force_j = -f_p * self.ps.density[object_j]*self.ps.m_V0
                torque_j = ti.math.cross(x_j - center_of_mass_j, force_j)
                self.ps.rigid_body_forces[object_j] += force_j
                self.ps.rigid_body_torques[object_j] += torque_j
                self.ps.pressure_force[p_j] += force_j
                    
        if self.ps.material[p_j] == self.ps.material_grain:
            dpi = self.ps.pressure[p_i] / (self.ps.density[p_i] ** 2)
            effective_density = 0.5 * (self.ps.density[p_i] + self.ps.density[p_j])  # or use fluid_density
            pressure_force = -1 * effective_density * self.ps.m_V[p_j] * dpi * self.poly6_kernel_derivative(x_i - x_j)
            ret += pressure_force


    @ti.kernel
    def compute_pressure_forces(self):
        for p_i in ti.grouped(self.ps.x):
            if self.ps.active[p_i] and self.ps.material[p_i] != self.ps.material_fluid:
                continue
            self.ps.density[p_i] = ti.max(self.ps.density[p_i], self.fluid_density)
            self.ps.pressure[p_i] = self.stiffness * (ti.pow(self.ps.density[p_i] / self.fluid_density, self.exponent) - 1.0)
        for p_i in ti.grouped(self.ps.x):
            if self.ps.active[p_i] and self.ps.material[p_i] == self.ps.material_fluid:
                dv = ti.Vector([0.0 for _ in range(self.ps.dim)])
                self.ps.for_all_neighbors(p_i, self.compute_pressure_forces_task, dv)
                self.ps.acceleration[p_i] += dv
                
    @ti.func
    def compute_effective_viscosity_newtonian(self):
        return self.viscosity

    @ti.func
    def compute_effective_viscosity(self, shear_rate):
        mu_eff = self.compute_effective_viscosity_newtonian()
        return mu_eff
    @ti.func
    def compute_non_pressure_forces_task(self, p_i, p_j, ret: ti.template()):
        x_i = self.ps.x[p_i]
        x_j = self.ps.x[p_j]
        r_ij = x_i - x_j
        r_norm = r_ij.norm() + 1e-5
        if self.ps.material[p_j] == self.ps.material_fluid:
            shear_rate = 0.0
            if self.viscosity_model != 0:  
                shear_rate = self.compute_shear_rate(p_i, p_j) + 1e-6
            mu_eff = self.compute_effective_viscosity(shear_rate)
            v_xy = (self.ps.v[p_i] - self.ps.v[p_j]).dot(r_ij)
            f_v = mu_eff * (self.ps.m[p_j] / (self.ps.density[p_j])) * v_xy / (
                r_norm**2 + 0.01 * self.ps.support_radius**2) * self.poly6_kernel_derivative(r_ij)
            ret += f_v
        if self.ps.material[p_j] == self.ps.material_fluid:
            diameter2 = self.ps.particle_diameter * self.ps.particle_diameter
            if r_norm > self.ps.particle_diameter**2:
                ret -= self.surface_tension / self.ps.m[p_i] * self.ps.m[p_j] * r_ij * self.poly6_kernel(r_norm)
            else:
                ret -= self.surface_tension / self.ps.m[p_i] * self.ps.m[p_j] * r_ij * self.poly6_kernel(ti.Vector([self.ps.particle_diameter, 0.0, 0.0]).norm())
                
        if self.ps.material[p_j] == self.ps.material_solid :
            boundary_viscosity = 0
            v_xy = (self.ps.v[p_i] - self.ps.v[p_j]).dot(r_ij)
            f_v = boundary_viscosity * (self.fluid_density * self.ps.m_V[p_j] / (self.ps.density[p_i])) * v_xy / (
                r_norm**2 + 0.01 * self.ps.support_radius**2) * self.poly6_kernel_derivative(r_ij)
            ret += f_v
            if self.ps.active[p_i] and self.ps.is_dynamic_rigid_body(p_j):
                object_j = self.ps.object_id[p_j]
                center_of_mass_j = self.ps.rigid_body_centers_of_mass[object_j]
                force_j = -f_v * self.ps.density[object_j] * self.ps.m_V0
                torque_j = ti.math.cross(x_j - center_of_mass_j, force_j)
                self.ps.rigid_body_forces[object_j] += force_j
                self.ps.rigid_body_torques[object_j] += torque_j
                
        if self.ps.material[p_j] == self.ps.material_grain:
            drag_force = self.dem_drag_coefficient * self.ps.m[p_j] * (self.ps.v[p_j] - self.ps.v[p_i])
            ret += drag_force
    @ti.kernel
    def compute_non_pressure_forces(self):
        for p_i in ti.grouped(self.ps.x):
            if self.ps.active[p_i] and self.ps.is_static_rigid_body(p_i):
                self.ps.acceleration[p_i].fill(0.0)
                continue
            d_v = ti.Vector(self.g)
            self.ps.acceleration[p_i] = d_v
            if self.ps.active[p_i] and self.ps.material[p_i] == self.ps.material_fluid:
                self.ps.for_all_neighbors(p_i, self.compute_non_pressure_forces_task, d_v)
                self.ps.acceleration[p_i] = d_v

    @ti.kernel
    def advect_fluid(self):
        for p_i in ti.grouped(self.ps.x):
            if self.ps.active[p_i] and self.ps.material[p_i] == self.ps.material_fluid:
                    self.ps.v[p_i] += self.dt[None] * self.ps.acceleration[p_i]
                    self.ps.x[p_i] += self.dt[None] * self.ps.v[p_i]
                    
    @ti.kernel
    def advect(self):
        # Symplectic Euler
        for p_i in ti.grouped(self.ps.x):
            if self.ps.is_dynamic[p_i]:
                self.ps.v[p_i] += self.dt[None] * self.ps.acceleration[p_i]
                self.ps.x[p_i] += self.dt[None] * self.ps.v[p_i]
#-------------------------------------------------------Curvature compute-------------------------------------------------------------------------------------------------------------------#
    @ti.func
    def W(self, r_ij):
        res = ti.cast(0.0, ti.f32)
        h = self.ps. support_radius
        norm_r_ij = r_ij.norm()
        if norm_r_ij < h:
            res = 15.0 / (np.pi * h**3) * (1 - norm_r_ij / h)**3
        return res
    @ti.func
    def gradient_W(self, r_ij):
        grad = ti.Vector([0.0, 0.0, 0.0])
        h = self.ps.support_radius
        norm_r_ij = r_ij.norm()
        if 0 < norm_r_ij < h:
            grad = -45.0 / (np.pi * h**6) * (h - norm_r_ij)**2 * r_ij / norm_r_ij
        return grad
    @ti.func
    def compute_curvature_task(self, p_i, p_j, ret: ti.template()):
        x_i = self.ps.x[p_i]
        x_j = self.ps.x[p_j]
        n_i = self.ps.normals[p_i]
        n_j = self.ps.normals[p_j]
        r_ij = x_i - x_j
        norm_r_ij = r_ij.norm()
        curvature_ij = (1 - n_i.dot(n_j)) * self.W(r_ij)
        if r_ij.normalized().dot(n_i) < 0: 
            ret += curvature_ij
                    
    @ti.kernel
    def compute_curvature_using_position(self):
        for p_i in ti.grouped(self.ps.x):
            if self.ps.material[p_i] == self.ps.material_fluid and self.ps.normal_magnitude[p_i] > self.normal_threshold:
                curvature = 0.0
                self.ps.for_all_neighbors_ww(p_i, self.compute_curvature_task, curvature)
                self.ps.curvature[p_i] = curvature
                if curvature > self.ps.max_curvature[None]:
                    self.ps.max_curvature[None] = curvature
            else:
                self.ps.curvature[p_i] = 0.0  
                    
    @ti.func
    def compute_color_field_gradient_task(self, p_i, p_j, ret: ti.template()):
        x_i = self.ps.x[p_i]
        x_j = self.ps.x[p_j]
        r_ij = x_i - x_j
        grad_W = self.gradient_W(r_ij)
        c_j = 1.0 if self.ps.material[p_j] == self.ps.material_fluid else 0.0
        ret += self.ps.m_V[p_j] * c_j * grad_W
    @ti.kernel
    def compute_color_field_gradient(self):
        for p_i in ti.grouped(self.ps.x):
            if self.ps.material[p_i] == self.ps.material_fluid:
                color_grad = ti.Vector([0.0, 0.0, 0.0])
                self.ps.for_all_neighbors(p_i, self.compute_color_field_gradient_task, color_grad)
                self.ps.normals[p_i] = color_grad
                
    @ti.kernel
    def update_surface_normals(self):
        for p_i in ti.grouped(self.ps.x):
            normal = self.ps.normals[p_i]
            normal_magnitude = normal.norm()
            self.ps.normal_magnitude[p_i] = normal_magnitude 
            if normal_magnitude > self.normal_threshold:
                self.ps.normals[p_i] = normal / normal_magnitude 
            else:
                self.ps.normals[p_i] = ti.Vector([0.0, 0.0, 0.0])
    @ti.func
    def smooth_normals_task(self, p_i, p_j, normal_sum_and_count: ti.template()):
        if self.ps.normal_magnitude[p_j] > self.normal_threshold:
            for k in ti.static(range(3)):
                normal_sum_and_count[None][k] += self.ps.normals[p_j][k]
            normal_sum_and_count[None][3] += 1.0

    @ti.kernel
    def smooth_normals(self):
        for p_i in ti.grouped(self.ps.x):
            if self.ps.normal_magnitude[p_i] > self.normal_threshold:
                self.ps.for_all_neighbors(p_i, self.smooth_normals_task, self.normal_sum_and_count)
                count = self.normal_sum_and_count[None][3]
                if count > 0:
                    normal_sum = ti.Vector([self.normal_sum_and_count[None][0], self.normal_sum_and_count[None][1], self.normal_sum_and_count[None][2]])
                    smoothed_normal = normal_sum / count
                    self.ps.normals[p_i] = smoothed_normal.normalized()
                else:
                    self.ps.normals[p_i] = ti.Vector([0.0, 0.0, 0.0])
            else:
                self.ps.normals[p_i] = ti.Vector([0.0, 0.0, 0.0])
    @ti.func
    def compute_shear_rate(self, p_i, p_j):
        v_ij = self.ps.v[p_i] - self.ps.v[p_j] 
        r_ij = self.ps.x[p_i] - self.ps.x[p_j]  
        r_norm = r_ij.norm() + 1e-5  
        return abs(v_ij.dot(r_ij)) / r_norm



#-----------------------------------------------------------DEM SOLVER------------------------------------------------------------------------------------#
    @ti.func
    def compute_dem_forces_task(self, p_i, p_j, ret: ti.template()):
        x_i = self.ps.x[p_i]
        x_j = self.ps.x[p_j]
        r_ij = x_i - x_j
        r_norm = r_ij.norm() + 1e-5
        if self.ps.material[p_j] == self.ps.material_grain:

            xi = self.ps.x[p_i]
            xj = self.ps.x[p_j]
            vi = self.ps.v[p_i]
            vj = self.ps.v[p_j]
            r_ij = xi - xj
            dist = r_ij.norm()
            n = r_ij / dist
            rel_v = vi - vj
            overlap = (self.ps.r[p_i] + self.ps.r[p_j])- dist
            if overlap > 0:
                normal = r_ij / dist
                normal_force = self.dem_stiffness * overlap * normal
                normal_velocity = (self.ps.v[p_j] - self.ps.v[p_i]).dot(normal)
                reduced_mass = (self.ps.m[p_i] * self.ps.m[p_j]) / (self.ps.m[p_i] + self.ps.m[p_j])
                damping_factor = 1.0 / ti.sqrt(1.0 + (math.pi / ti.log(self.dem_damping_coefficient))**2)
                damping = 2.0 * damping_factor * ti.sqrt(self.dem_stiffness * reduced_mass)
                damping_force = damping * normal_velocity * normal
                tangential_velocity = rel_v - normal_velocity * normal
                tangential_speed = tangential_velocity.norm()
                friction_force = ti.Vector([0.0, 0.0, 0.0])
                if tangential_speed > 1e-5:
                    friction_direction = -tangential_velocity / tangential_speed
                    friction_factor = self.dem_friction_coefficient * normal_force.norm()
                    friction_force = friction_factor * friction_direction
                else:
                    friction_force = ti.Vector([0.0, 0.0, 0.0])
                force_total = normal_force + damping_force + friction_force
                ret += force_total
        if self.ps.material[p_j] == self.ps.material_fluid:
            buoyancy_force = -(self.ps.density[p_j] - self.ps.density[p_i]) * ti.Vector(self.g) * self.ps.m_V[p_j]
            rel_v = self.ps.v[p_j] - self.ps.v[p_i]
            rel_speed = rel_v.norm()
            area = math.pi * self.ps.r[p_i] ** 2
            fluid_density = self.ps.density[p_j]
            drag_force = 0.5* self.dem_drag_coefficient * area * fluid_density * rel_speed * rel_v
            ret += drag_force + buoyancy_force
        if self.ps.material[p_j] == self.ps.material_solid :
            xi = self.ps.x[p_i]
            xj = self.ps.x[p_j]
            vi = self.ps.v[p_i]
            vj = self.ps.v[p_j]
            r_ij = xi - xj
            dist = r_ij.norm()
            normal = r_ij / dist
            overlap = (self.ps.r[p_i] + self.ps.r[p_j]) - dist
            if overlap > 0:
                normal_force = self.dem_stiffness * overlap * normal
                normal_velocity = (vj - vi).dot(normal)
                damping_force = self.dem_damping_coefficient * normal_velocity * normal
                tangential_velocity = (vj - vi) - normal_velocity * normal
                friction_force = self.dem_friction_coefficient * normal_force.norm() * (-tangential_velocity.normalized())
                total_force = normal_force + damping_force + friction_force
                ret += total_force
                if self.ps.is_dynamic_rigid_body(p_j):
                    object_j = self.ps.object_id[p_j]
                    center_of_mass_j = self.ps.rigid_body_centers_of_mass[object_j]
                    force_j = -total_force * self.ps.density[object_j] * self.ps.m_V0
                    torque_j = ti.math.cross(x_j - center_of_mass_j, force_j)
                    ti.atomic_add(self.ps.rigid_body_forces[object_j], force_j)
                    ti.atomic_add(self.ps.rigid_body_torques[object_j], torque_j)

    @ti.kernel
    def compute_dem_forces(self):
        for p_i in ti.grouped(self.ps.x):
            if self.ps.active[p_i] and self.ps.material[p_i] == self.ps.material_grain:
                contact_force = ti.Vector([0.0, 0.0, 0.0])
                self.ps.for_all_neighbors(p_i, self.compute_dem_forces_task, contact_force)
                self.ps.acceleration[p_i] += contact_force / (self.ps.m[p_i])
    
    @ti.kernel
    def advect_gravity(self):
        for p_i in ti.grouped(self.ps.x):
            if self.ps.material[p_i] == self.ps.material_grain:
                self.ps.acceleration[p_i] = ti.Vector(self.g)

    @ti.kernel
    def advect_granular(self):
        for p_i in ti.grouped(self.ps.x):
            if self.ps.material[p_i] == self.ps.material_grain:
                self.ps.v[p_i] += 0.5*self.dt[None] * self.ps.acceleration[p_i]
                self.ps.x[p_i] += self.dt[None] * self.ps.v[p_i] + 0.5 * self.dt[None]**2 * self.ps.acceleration[p_i]
                self.ps.v[p_i] *= 0.99995

#-----------------------------------------------------------------------------------------------------#   
    def substep(self):       
        if self.ps.num_fluid_blocks > 0 :
            self.compute_densities()
            self.compute_non_pressure_forces()
            self.compute_pressure_forces()
            self.advect_fluid()
            if self.export_fluid_velocity:
                self.compute_color_field_gradient()
                self.update_surface_normals() 
                self.compute_curvature_using_position()
        if self.ps.num_grain_objects > 0 :
            self.advect_gravity()
            self.compute_dem_forces()
            self.advect_granular()
  



