# Doriflow Engine - Fluid Simulation for Blender 3D
# Copyright (C) 2024 Doriflow Team
# This software is licensed under the Creative Commons Attribution-NonCommercial 4.0 International License (CC BY-NC 4.0).

# You are free to:
# -Share: Copy and redistribute the material in any medium or format.
# -Adapt: Remix, transform, and build upon the material.

# UNDER THE FOLOWING TERMS:
# -Attribution: You must give appropriate credit, provide a link to the license, and indicate if changes were made.
# -Appropriate credit should include the following:
#   -The original author's name: Doriflow Team
#   -A link to the original source (if applicable).
#   -A link to the full license: https://creativecommons.org/licenses/by-nc/4.0/.
#   -A clear indication of any changes made, such as: "This material has been modified."
# NonCommercial: You may not use the material for commercial purposes.
# Disclaimer:
# -This simulation engine is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. In no event shall the authors be liable for any claim, damages, or other liability arising from the use of this software.

# For more details, refer to the full license text at:
# https://creativecommons.org/licenses/by-nc/4.0/.

#-----------------------------------------------------------------------------------------------------------------------#
import json
import atexit
import bpy
from bpy.types import Operator
from bpy.props import StringProperty, IntProperty
import os
import sys
import subprocess
import shutil
from ..utils import use_blender_env
import tempfile


def pause_fluid_simulation(compute_fluid_props, drop_file=True):
    compute_fluid_props.is_compute_fluid_paused = True
    if drop_file:
        command_file_path = os.path.join(compute_fluid_props.output_directory_path, 'pause.json')
        with open(command_file_path, 'w') as f:
            json.dump({}, f)


def stop_fluid_simulation(compute_fluid_props, drop_file=True):
    compute_fluid_props.is_compute_fluid_stopped = True
    if drop_file:
        command_file_path = os.path.join(compute_fluid_props.output_directory_path, 'stop.json')
        with open(command_file_path, 'w') as f:
            json.dump({}, f)

def delete_voxelized_particles(self):
    objects_to_delete = [obj for obj in bpy.data.objects if obj.name.endswith('_initial_voxelized_particles')]
    if objects_to_delete:
        for obj in objects_to_delete:
            bpy.data.objects.remove(obj, do_unlink=True)
        self.report({'INFO'}, f"Deleted {len(objects_to_delete)} voxelized particle objects.")
    else:
        self.report({'INFO'}, "No voxelized particle objects found to delete.")

def unhide_objects():
    for obj in bpy.data.objects:
        if hasattr(obj, "doriflow") and hasattr(obj.doriflow, "object_type"):
            obj.hide_set(False)
            obj.hide_viewport = False
    
class DORIFLOW_OT_compute_fluidSimulation(Operator):
    bl_idname = "doriflow.compute_fluid_simulation"
    bl_label = "Compute Doriflow Fluid Simulation"
    bl_options = {'REGISTER'}
    bl_description = "Start the Doriflow simulation"

    def _get_compute_fluid_props(self):
        return bpy.context.scene.doriflow.compute_fluid
    def _reset_compute_fluid(self, context):
        compute_fluid_props = self._get_compute_fluid_props()
        compute_fluid_props.is_simulation_running = True
        compute_fluid_props.is_compute_fluid_initialized = False
        compute_fluid_props.is_compute_fluid_paused = False
        compute_fluid_props.is_compute_fluid_stopped = False
    @classmethod
    def poll(cls, context):
        compute_fluid_props = bpy.context.scene.doriflow.compute_fluid
        compute_mesh_props = bpy.context.scene.doriflow.compute_mesh
        if compute_fluid_props is None:
            return False
        elif compute_fluid_props.is_one_button_compute_running:
            return (not compute_fluid_props.is_simulation_running) & (not compute_mesh_props.is_compute_mesh_running)
        else:
            return not compute_fluid_props.is_simulation_running

    def create_material(self):
        mat_name = "DF.Fluid_Material"
        if mat_name not in bpy.data.materials:
            mat = bpy.data.materials.new(name=mat_name)
            mat.use_nodes = True
            nodes = mat.node_tree.nodes
            nodes.clear() 
            bsdf_node = nodes.new(type='ShaderNodeBsdfPrincipled')
            material_output = nodes.new(type='ShaderNodeOutputMaterial')
            links = mat.node_tree.links
            links.new(bsdf_node.outputs['BSDF'], material_output.inputs['Surface'])
            color = (0x8A / 0xFF, 0xD1 / 0xFF, 0xFF / 0xFF, 0.8)  
            bsdf_node.inputs['Base Color'].default_value = color
            bsdf_node.inputs['Alpha'].default_value = 1
            bsdf_node.inputs['Transmission Weight'].default_value = 1
            bsdf_node.inputs['Roughness'].default_value = 0.0
        else:
            mat = bpy.data.materials[mat_name]
        return mat
    def modal(self, context, event):
        compute_fluid_props = self._get_compute_fluid_props()
        if not self.is_cmd_launched and compute_fluid_props.is_compute_fluid_stopped:
            self.cancel(context)
            return {'FINISHED'}
        if not self.is_cmd_launched:
            try:
                blender_file_path = bpy.data.filepath
                blender_file_dir = os.path.dirname(blender_file_path)
                blender_file_name = ".".join(os.path.splitext(os.path.basename(blender_file_path))[:-1])
                cache_folder_name = f"{blender_file_name}_cache"
                cache_folder_path = os.path.join(blender_file_dir, cache_folder_name)
                if os.path.exists(cache_folder_path):
                    for filename in os.listdir(cache_folder_path):
                        file_path = os.path.join(cache_folder_path, filename)
                        try:
                            if os.path.isfile(file_path) or os.path.islink(file_path):
                                os.unlink(file_path)
                            elif os.path.isdir(file_path):
                                shutil.rmtree(file_path)
                        except Exception as e:
                            print('Failed to delete %s. Reason: %s' % (file_path, e))
                else:
                    os.makedirs(cache_folder_path)
                json_file_name = f"{blender_file_name}_initial_condition.json"
                json_file_path = os.path.join(cache_folder_path, json_file_name)
                self._cache_dir = cache_folder_path
                compute_fluid_props.output_directory_path = self._cache_dir
                status_file_path = os.path.join(self._cache_dir, 'status.json')
                command_file_path = os.path.join(self._cache_dir, 'stop.json')
                if os.path.exists(status_file_path):
                    os.remove(status_file_path)
                if os.path.exists(command_file_path):
                    os.remove(command_file_path)
                domain_objects = [obj for obj in bpy.data.objects if obj.doriflow.object_type == 'TYPE_DOMAIN']
                obstacle_objects = [obj for obj in bpy.data.objects if obj.doriflow.object_type == 'TYPE_OBSTACLE']
                fluid_objects = [obj for obj in bpy.data.objects if obj.doriflow.object_type == 'TYPE_FLUID']
                grain_objects = [obj for obj in bpy.data.objects if obj.doriflow.object_type == 'TYPE_GRAIN']

                
                if len(domain_objects) >0:
                    bpy.ops.doriflow.export_domain()
                if len(obstacle_objects) >0:
                    bpy.ops.doriflow.export_obstacle()
                if len(fluid_objects) >0:
                    bpy.ops.doriflow.export_fluid()
                if len(grain_objects) >0:
                    bpy.ops.doriflow.export_grain()
                bpy.context.view_layer
                
                bpy.ops.doriflow.export_data()

                bpy.ops.object.select_all(action='DESELECT')
               
                if domain_objects:
                    domain_object = domain_objects[0]  
                    bpy.context.view_layer.objects.active = domain_object 
                    domain_object.select_set(True)
                operators_directory = os.path.dirname(__file__)
                addon_root_directory = os.path.dirname(operators_directory)
                compute_script_directory = os.path.join(addon_root_directory, "Doriflow_modules")
                compute_script_path = os.path.join(compute_script_directory, "compute.py")
                if not os.path.exists(compute_script_path):
                    self.report({'ERROR'}, "compute_fluid script not found.")
                    self.cancel(context)
                    return {'CANCELLED'}
                packages_path = os.path.join(addon_root_directory, "packages")

                if not os.path.exists(packages_path) or not any(os.path.isdir(os.path.join(packages_path, d)) for d in os.listdir(packages_path)):
                    self.report({'ERROR'}, "You have not finalized the solver installation. Please go to Edit > Preferences > Add-ons > Doriflow Addon, expand the panel, and click 'Finalize Solver Dependencies'.")
                    return {'CANCELLED'}
                use_blender_env()
                if sys.platform == 'win32':
                    batch_file_path = os.path.join(compute_script_directory, "run_script.bat")
                    blender_python_exe = os.path.join(os.path.dirname(sys.executable), "python.exe")
                    cmd = [batch_file_path, blender_python_exe, compute_script_path, json_file_path]
                    env = os.environ.copy()
                    packages_path = os.path.join(addon_root_directory, "packages")
                    subfolders = [os.path.join(packages_path, d) for d in os.listdir(packages_path) if os.path.isdir(os.path.join(packages_path, d))]
                    packages_path_env = ";".join(subfolders) + ";" + env.get("PYTHONPATH", "")
                    env["PYTHONPATH"] = packages_path_env  # Update the environment variable
                    subprocess.Popen(cmd, creationflags=subprocess.CREATE_NEW_CONSOLE, cwd=compute_script_directory, env=env)
                    atexit.register(self.terminate_subprocess_on_exit)
                    self.report({'INFO'}, "Simulation started in external process.")
                    bpy.ops.wm.monitor_simulation_progress()
                #check for mac os
                elif sys.platform == 'darwin':
                    batch_file_path_mac = os.path.join(compute_script_directory, "run_script_mac.sh")
                    executable_path = os.path.abspath(sys.executable)
                    packages_path = os.path.join(addon_root_directory, "packages")
                    subfolders = [os.path.join(packages_path, d) for d in os.listdir(packages_path) if os.path.isdir(os.path.join(packages_path, d))]
                    packages_path_env = ":".join(subfolders) + ":$PYTHONPATH"
                    command = f'''
                    tell application "Terminal"
                        do script "export PYTHONPATH='{packages_path_env}'; cd '{compute_script_directory}'; chmod +x ./run_script_mac.sh; ./run_script_mac.sh '{executable_path}' '{compute_script_path}' '{json_file_path}'"
                        activate
                    end tell
                    '''
                    subprocess.Popen(['osascript', '-e', command])
                    atexit.register(self.terminate_subprocess_on_exit)
                    self.report({'INFO'}, "Simulation started in external process.")
                    bpy.ops.wm.monitor_simulation_progress()
            except Exception as e:
                self.report({'ERROR'}, f"Failed to start simulation: {e}")
                self.cancel(context)
                return {'CANCELLED'}
            self.is_cmd_launched = True
        if self.is_cmd_finished:
            self.cancel(context)
            return {'FINISHED'}
        if event.type == 'TIMER' and not self.is_updating_status:
            self.is_updating_status = True
            self._update_status(context)
            self.is_updating_status = False
        return {'PASS_THROUGH'}
    def terminate_subprocess_on_exit(self):
        if self.subprocess:
            try:
                self.subprocess.terminate()  
                self.subprocess.wait(timeout=5)  
            except Exception as e:
                print(f"Failed to terminate subprocess on exit: {e}")
    def _update_status(self, context): 
        compute_fluid_props = self._get_compute_fluid_props()
        try:
            finished_path = os.path.join(self._cache_dir, 'finish.txt')
            if os.path.exists(finished_path) or compute_fluid_props.is_compute_fluid_stopped:
                if os.path.exists(finished_path):
                    os.remove(finished_path)
                self.is_cmd_finished = True
                compute_fluid_props.is_safe_to_exit = True
                error_path = os.path.join(self._cache_dir, 'error.txt')
                if os.path.exists(error_path):
                    with open(error_path, 'r') as f:
                        error_message = f.read()
                    os.remove(error_path)
                    bpy.ops.doriflow.display_error(
                        'INVOKE_DEFAULT',
                        error_message="Error Computing Doriflow Simulation",
                        error_description=error_message,
                        popup_width=400
                        )
            status_file_path = os.path.join(self._cache_dir, 'status.json')
            if os.path.exists(status_file_path):
                with open(status_file_path, 'r') as f:
                    status_dict = json.load(f)
                compute_fluid_props.is_compute_fluid_initialized = status_dict['initialized']
                compute_fluid_props.num_total_frames = status_dict['total_frames']
                compute_fluid_props.num_compute_fluid_frames = status_dict['cnt_frame']
                compute_fluid_props.num_compute_fluidd_vtks = status_dict['cnt_vtk']
        except json.JSONDecodeError as e:
            print(str(e))
        except Exception as e:
            print(str(e))
            raise e
    def execute(self, context):
        self.timer = None
        self.is_cmd_launched = False
        self.is_cmd_finished = False
        self.is_updating_status = False
        self._cache_dir = None
        unhide_objects()
        delete_voxelized_particles(self)
        bpy.context.scene.frame_set(1)
        blender_file_path = bpy.data.filepath
        objects_to_delete = [obj for obj in bpy.data.objects if obj.name.endswith('_initial_voxelized_particles')]
        if objects_to_delete:
            self.report(
                {'ERROR'},
                "The following voxelized particle objects need to be deleted before running the simulation:\n" +
                "\n".join([obj.name for obj in objects_to_delete])
            )
            return {'CANCELLED'}
        if not blender_file_path:
            self.report({'ERROR'}, "Blender file is not saved. Please save the file before running the simulation.")
            return {'CANCELLED'}
        self._reset_compute_fluid(context)
        context.window_manager.modal_handler_add(self)
        self.timer = context.window_manager.event_timer_add(0.1, window=context.window)
        return {'RUNNING_MODAL'}
    def cancel(self, context):
        if self.timer:
            context.window_manager.event_timer_remove(self.timer)
            self.timer = None
        compute_fluid_props = self._get_compute_fluid_props()
        if compute_fluid_props is None:
            return
        compute_fluid_props.is_simulation_running = False
        compute_fluid_props.is_compute_fluid_paused = False
        compute_fluid_props.is_compute_fluid_stopped = False
class DORIFLOW_OT_Pausecompute_fluidSimulation(bpy.types.Operator):
    bl_idname = "doriflow.pause_compute_fluid_simulation"
    bl_label = "Pause Compute Fluid Simulation"
    bl_description = "Pause compute fluid simulation"

    @classmethod
    def poll(cls, context):
        compute_fluid_props = bpy.context.scene.doriflow.compute_fluid
        if compute_fluid_props is None:
            return False
        elif compute_fluid_props.is_one_button_compute_running:
            return False
        return not compute_fluid_props.is_compute_fluid_paused
    def execute(self, context):
        compute_fluid_props = bpy.context.scene.doriflow.compute_fluid
        if compute_fluid_props is None:
            return {'CANCELLED'}
        pause_fluid_simulation(compute_fluid_props)
        return {'FINISHED'}
class DORIFLOW_OT_Stopcompute_fluidSimulation(bpy.types.Operator):
    bl_idname = "doriflow.stop_compute_fluid_simulation"
    bl_label = "Stop compute_fluid"
    bl_description = "Stop simulation compute_fluid and reset to initial state"
    @classmethod
    def poll(cls, context):
        compute_fluid_props = bpy.context.scene.doriflow.compute_fluid
        if compute_fluid_props is None:
            return False
        elif compute_fluid_props.is_one_button_compute_running:
            return False
        return compute_fluid_props.is_compute_fluid_paused
    def execute(self, context):
        compute_fluid_props = bpy.context.scene.doriflow.compute_fluid
        if compute_fluid_props is None:
            return {'CANCELLED'}
        stop_fluid_simulation(compute_fluid_props)
        return {'FINISHED'}
    def invoke(self, context, event):
        return context.window_manager.invoke_confirm(self, event)
class DORIFLOW_OT_Resumecompute_fluidSimulation(bpy.types.Operator):
    bl_idname = "doriflow.resume_compute_fluid_simulation"
    bl_label = "Resume compute_fluid"
    bl_description = "Resume simulation compute_fluid from the pause"
    @classmethod
    def poll(cls, context):
        compute_fluid_props = bpy.context.scene.doriflow.compute_fluid
        if compute_fluid_props is None:
            return False
        elif compute_fluid_props.is_one_button_compute_running:
            return False
        return compute_fluid_props.is_compute_fluid_paused
    def execute(self, context):
        compute_fluid_props = bpy.context.scene.doriflow.compute_fluid
        if compute_fluid_props is None:
            return {'CANCELLED'}
        compute_fluid_props.is_compute_fluid_paused = False
        command_file_path = os.path.join(compute_fluid_props.output_directory_path, 'continue.json')
        with open(command_file_path, 'w') as f:
            json.dump({}, f)
        return {'FINISHED'}
class DORIFLOW_OT_MonitorSimulationProgress(Operator):
    bl_idname = "wm.monitor_simulation_progress"
    bl_label = "Monitor Simulation Progress"
    _timer = None
    def modal(self, context, event):
        blender_file_path = bpy.data.filepath
        if not blender_file_path:
            self.report({'ERROR'}, "Blender file is not saved. Please save the file before running the export.")
            return {'CANCELLED'}
        blender_file_dir = os.path.dirname(blender_file_path)
        blender_file_name = os.path.splitext(os.path.basename(blender_file_path))[0]
        cache_folder_name = f"{blender_file_name}_cache"
        cache_folder_path = os.path.join(blender_file_dir, cache_folder_name)
        sim_output_folder_name = f"{blender_file_name}_output"
        if event.type == 'TIMER':
            progress_file_path = os.path.join(cache_folder_path,sim_output_folder_name, "simulation_progress.txt")
            try:
                with open(progress_file_path, "r") as file:
                    progress = int(file.read().strip())
                    context.scene.simulation_progress = progress
            except Exception as e:
                print(f"Error reading simulation progress: {e}")
            if hasattr(context, 'area') and context.area is not None:
                context.area.tag_redraw()
            return {'PASS_THROUGH'}
        return {'PASS_THROUGH'}
    def execute(self, context):
        wm = context.window_manager
        self._timer = wm.event_timer_add(1.0, window=context.window) 
        wm.modal_handler_add(self)
        return {'RUNNING_MODAL'}
    def invoke(self, context, event):
        self.execute(context)
        return {'RUNNING_MODAL'}
def register():
    bpy.utils.register_class(DORIFLOW_OT_compute_fluidSimulation)
    bpy.utils.register_class(DORIFLOW_OT_Pausecompute_fluidSimulation)
    bpy.utils.register_class(DORIFLOW_OT_Stopcompute_fluidSimulation)
    bpy.utils.register_class(DORIFLOW_OT_Resumecompute_fluidSimulation)
    bpy.utils.register_class(DORIFLOW_OT_MonitorSimulationProgress)
    bpy.types.Scene.simulation_progress = IntProperty(
        name="Simulation Progress",
        description="Current simulation progress",
        default=0,
    )
 
def unregister():
    bpy.utils.unregister_class(DORIFLOW_OT_Resumecompute_fluidSimulation)
    bpy.utils.unregister_class(DORIFLOW_OT_Stopcompute_fluidSimulation)
    bpy.utils.unregister_class(DORIFLOW_OT_Pausecompute_fluidSimulation)
    bpy.utils.unregister_class(DORIFLOW_OT_compute_fluidSimulation)
    bpy.utils.unregister_class(DORIFLOW_OT_MonitorSimulationProgress)
    del bpy.types.Scene.simulation_progress
