# Doriflow Engine - Fluid Simulation for Blender 3D
# Copyright (C) 2024 Doriflow Team
# This software is licensed under the Creative Commons Attribution-NonCommercial 4.0 International License (CC BY-NC 4.0).

# You are free to:
# -Share: Copy and redistribute the material in any medium or format.
# -Adapt: Remix, transform, and build upon the material.

# UNDER THE FOLOWING TERMS:
# -Attribution: You must give appropriate credit, provide a link to the license, and indicate if changes were made.
# -Appropriate credit should include the following:
#   -The original author's name: Doriflow Team
#   -A link to the original source (if applicable).
#   -A link to the full license: https://creativecommons.org/licenses/by-nc/4.0/.
#   -A clear indication of any changes made, such as: "This material has been modified."
# NonCommercial: You may not use the material for commercial purposes.
# Disclaimer:
# -This simulation engine is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. In no event shall the authors be liable for any claim, damages, or other liability arising from the use of this software.

# For more details, refer to the full license text at:
# https://creativecommons.org/licenses/by-nc/4.0/.

#-----------------------------------------------------------------------------------------------------------------------#
import bpy
import os
import shutil
import subprocess

class RENDER_OT_manual_frame_advance(bpy.types.Operator):
    bl_idname = "doriflow.custom_render_ww"
    bl_label = "Manual Frame Advance Render"
    bl_description = "Render the Doriflow animation frame by frame based on user-selected range"
    _timer = None

    def execute(self, context):
        render_start_frame = context.scene.doriflow.render_start_frame
        render_end_frame = context.scene.doriflow.render_end_frame
        if render_start_frame > render_end_frame:
            self.report({'ERROR'}, "Start frame must be less than or equal to end frame.")
            return {'CANCELLED'}
        render_output_path = os.path.abspath(context.scene.doriflow.render_output_path)
        if not os.path.isdir(render_output_path):
            try:
                os.makedirs(render_output_path, exist_ok=True)
            except Exception as e:
                self.report({'ERROR'}, f"Invalid output path: {render_output_path}\n{e}")
                return {'CANCELLED'}
        self.original_render_path = context.scene.render.filepath
        self.frame = render_start_frame
        context.window_manager.modal_handler_add(self)
        self._timer = context.window_manager.event_timer_add(0.3, window=context.window)
        return {'RUNNING_MODAL'}

    def modal(self, context, event):
        if event.type in {'ESC', 'RIGHTMOUSE'}:
            self.finish(context, cancel=True)
            return {'CANCELLED'}
        if event.type == 'TIMER':
            render_end_frame = context.scene.doriflow.render_end_frame
            if self.frame > render_end_frame:
                self.finish(context)
                return {'FINISHED'}
            context.scene.frame_set(self.frame)
            render_file_path = os.path.join(
                context.scene.doriflow.render_output_path,
                f"frame_{str(self.frame).zfill(4)}.png"
            )
            context.scene.render.filepath = render_file_path
            bpy.ops.render.render(write_still=True)
            self.frame += 1
        return {'PASS_THROUGH'}

    def finish(self, context, cancel=False):
        context.scene.render.filepath = self.original_render_path
        context.window_manager.event_timer_remove(self._timer)
        if cancel:
            self.report({'INFO'}, "Rendering cancelled by user.")
        else:
            self.report({'INFO'}, "Animation rendered frame by frame successfully.")

class DORIFLOW_OT_select_render_output_path(bpy.types.Operator):
    bl_idname = "doriflow.select_render_output_path"
    bl_label = "Select Render Output Path"
    bl_description = "Select the directory to save rendered frames"
    bl_options = {'REGISTER', 'UNDO'}

    filepath: bpy.props.StringProperty(subtype="DIR_PATH")
    def execute(self, context):
        selected_path = os.path.abspath(os.path.normpath(self.filepath))
        if not os.path.isdir(selected_path):
            self.report({'ERROR'}, f"Invalid directory selected: {selected_path}")
            return {'CANCELLED'}
        context.scene.doriflow.render_output_path = selected_path
        context.area.tag_redraw()
        self.report({'INFO'}, f"Output path set to: {selected_path}")
        return {'FINISHED'}
    def invoke(self, context, event):
        context.window_manager.fileselect_add(self)
        self.filepath = ""
        return {'RUNNING_MODAL'}


def register():
    bpy.utils.register_class(RENDER_OT_manual_frame_advance)
    bpy.utils.register_class(DORIFLOW_OT_select_render_output_path)


def unregister():
    bpy.utils.unregister_class(RENDER_OT_manual_frame_advance)
    bpy.utils.unregister_class(DORIFLOW_OT_select_render_output_path)


if __name__ == "__main__":
    register()
