# Doriflow Engine - Fluid Simulation for Blender 3D
# Copyright (C) 2024 Doriflow Team
# This software is licensed under the Creative Commons Attribution-NonCommercial 4.0 International License (CC BY-NC 4.0).

# You are free to:
# -Share: Copy and redistribute the material in any medium or format.
# -Adapt: Remix, transform, and build upon the material.

# UNDER THE FOLOWING TERMS:
# -Attribution: You must give appropriate credit, provide a link to the license, and indicate if changes were made.
# -Appropriate credit should include the following:
#   -The original author's name: Doriflow Team
#   -A link to the original source (if applicable).
#   -A link to the full license: https://creativecommons.org/licenses/by-nc/4.0/.
#   -A clear indication of any changes made, such as: "This material has been modified."
# NonCommercial: You may not use the material for commercial purposes.
# Disclaimer:
# -This simulation engine is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. In no event shall the authors be liable for any claim, damages, or other liability arising from the use of this software.

# For more details, refer to the full license text at:
# https://creativecommons.org/licenses/by-nc/4.0/.
#-----------------------------------------------------------------------------------------------------------------------#
import bpy
import os
import numpy as np


def update_mesh_as_pointcloud(mesh: bpy.types.Mesh, points, radii=None):
    mesh.clear_geometry()
    num_points = len(points)
    if num_points == 0:
        return

    mesh.vertices.add(num_points)

    if isinstance(points, np.ndarray):
        co_flat = points.ravel().tolist()
    else:
        co_flat = []
        for (x, y, z) in points:
            co_flat.extend((x, y, z))
    mesh.vertices.foreach_set("co", co_flat)
    if radii is not None:
        if "radius" not in mesh.attributes:
            mesh.attributes.new(name="radius", type='FLOAT', domain='POINT')
        mesh.attributes["radius"].data.foreach_set("value", radii.tolist())

    mesh.update()

def read_vtk_points_polydata_legacy_binary(file_path):
    points = None
    scalars_dict = {}
    vectors_dict = {}
    is_binary = False
    def read_ascii_line(f):
        line = f.readline()
        if not line:
            return None
        return line.decode('utf-8', errors='ignore').strip()
    def read_binary_data(f, count, dtype):
        raw_data = f.read(count)
        if len(raw_data) != count:
            raise ValueError("Unexpected EOF while reading binary data.")
        return np.frombuffer(raw_data, dtype=dtype)
    with open(file_path, 'rb') as f:
        while True:
            line = read_ascii_line(f)
            if not line:
                raise ValueError("Unexpected end of file while reading header.")
            if line.startswith("ASCII"):
                is_binary = False
                break
            elif line.startswith("BINARY"):
                is_binary = True
                break
        while True:
            line = read_ascii_line(f)
            if not line:
                break
            tokens = line.split()
            if not tokens:
                continue
            keyword = tokens[0].upper()
            if keyword == "POINTS":
                print("keyword",keyword)
                num_points = int(tokens[1])
                data_type = tokens[2].lower()
                if is_binary:
                    points = read_binary_data(f, num_points * 3 * 4, dtype=">f4").reshape((num_points, 3))
                else:
                    points = []
                    for _ in range(num_points):
                        line = read_ascii_line(f)
                        points.append(list(map(float, line.split())))
                    points = np.array(points, dtype=np.float32)
            elif keyword == "VERTICES":
                print
                num_vertices = int(tokens[1])
                size_vertices = int(tokens[2])
                if is_binary:
                    f.read(size_vertices * 4)  
                else:
                    for _ in range(num_vertices):
                        read_ascii_line(f)
            elif keyword == "POINT_DATA":
                num_points_in_point_data = int(tokens[1])
                if points is None:
                    raise ValueError("POINT_DATA found before POINTS.")
            elif keyword == "SCALARS":
                scalar_name = tokens[1]
                read_ascii_line(f)  # Skip LOOKUP_TABLE line
                if is_binary:
                    raw_scalars = read_binary_data(f, num_points_in_point_data * 4, dtype=">f4")
                    scalars_dict[scalar_name] = raw_scalars
                else:
                    scalars = []
                    for _ in range(num_points_in_point_data):
                        scalars.append(float(read_ascii_line(f)))
                    scalars_dict[scalar_name] = np.array(scalars, dtype=np.float32)
            elif keyword == "VECTORS":
                vector_name = tokens[1]
                if is_binary:
                    raw_vectors = read_binary_data(f, num_points_in_point_data * 3 * 4, dtype=">f4")
                    vectors_dict[vector_name] = raw_vectors.reshape((num_points_in_point_data, 3))
                else:
                    vectors = []
                    for _ in range(num_points_in_point_data):
                        vectors.append(list(map(float, read_ascii_line(f).split())))
                    vectors_dict[vector_name] = np.array(vectors, dtype=np.float32)
    if points is None:
        raise ValueError(f"No POINTS found in '{file_path}'. Is it a valid VTK file?")
    return points, scalars_dict, vectors_dict
class DORIFLOW_OT_ImportAllParticles(bpy.types.Operator):
    bl_idname = "doriflow.import_all_particles"
    bl_label = "Import Fluid and Inlet Particles"
    bl_options = {'REGISTER', 'UNDO'}
    bl_description = "Import fluid and inlet particles from a simulation cache folder."

    particle_positions = {'liquid': {}, 'gas': {}, 'inlet_liquid': {}, 'inlet_gas': {}, 'inlet_grain':{}, 'grain': {}}
    has_liquid = False
    has_gas = False
    has_inlet_liquid = False
    has_inlet_gas = False
    has_inlet_grain = False
    has_grain = False

    def load_point_cloud(self, file_path, extract_radius=False):
        try:
            points, scalars, _ = read_vtk_points_polydata_legacy_binary(file_path)
            radius = scalars.get("radius", None) if extract_radius else None
            return points, radius
        except ValueError as e:
            print(f"Skipping file '{file_path}': {e}")
            return np.array([]), None


    def preload_vtk_data(self, vtk_directory):
        if not os.path.exists(vtk_directory):
            print(f"Directory '{vtk_directory}' does not exist.")
            return
        self.has_liquid = any(f.startswith('liquid') and f.endswith('.vtk') for f in os.listdir(vtk_directory))
        self.has_gas = any(f.startswith('gas') and f.endswith('.vtk') for f in os.listdir(vtk_directory))
        self.has_inlet_liquid = any(f.startswith('inlet_liquid') and f.endswith('.vtk') for f in os.listdir(vtk_directory))
        self.has_inlet_gas = any(f.startswith('inlet_gas') and f.endswith('.vtk') for f in os.listdir(vtk_directory))
        self.has_inlet_grain = any(f.startswith('inlet_grain') and f.endswith('.vtk') for f in os.listdir(vtk_directory))
        self.has_grain = any(f.startswith('grain') and f.endswith('.vtk') for f in os.listdir(vtk_directory))
        if not self.has_liquid and not self.has_gas and not self.has_inlet_liquid and not self.has_inlet_gas and not self.has_grain:
            print("No fluid or inlet particle files found. Skipping import.")
            return

        frame_offset = 1
        scene = bpy.context.scene
        start_frame = scene.compute_mesh_properties.import_particles_start_frame
        end_frame = scene.compute_mesh_properties.import_particles_end_frame
        for vtk_file in sorted(os.listdir(vtk_directory)):
            if vtk_file.endswith('.vtk'):
                original_frame_number = int(vtk_file.split('_')[-1].split('.')[0])
                adjusted_frame_number = original_frame_number + frame_offset
                if start_frame <= adjusted_frame_number <= end_frame:
                    file_path = os.path.join(vtk_directory, vtk_file)
                    extract_radius = vtk_file.startswith("grain") or vtk_file.startswith("inlet_grain")
                    points, radius = self.load_point_cloud(file_path, extract_radius=extract_radius)
                    if vtk_file.startswith('liquid'):
                        self.particle_positions['liquid'][adjusted_frame_number] = points
                    elif vtk_file.startswith('gas'):
                        self.particle_positions['gas'][adjusted_frame_number] = points
                    elif vtk_file.startswith('inlet_liquid'):
                        self.particle_positions['inlet_liquid'][adjusted_frame_number] = points
                    elif vtk_file.startswith('inlet_gas'):
                        self.particle_positions['inlet_gas'][adjusted_frame_number] = points
                    elif vtk_file.startswith('inlet_grain'):
                        self.particle_positions['inlet_grain'][adjusted_frame_number] = (points, radius)
                    elif vtk_file.startswith('grain'):
                        self.particle_positions['grain'][adjusted_frame_number] = (points, radius)
        if start_frame <= 2 <= end_frame:
            if 2 in self.particle_positions['liquid']:
                self.particle_positions['liquid'][1] = self.particle_positions['liquid'][2]
            if 2 in self.particle_positions['gas']:
                self.particle_positions['gas'][1] = self.particle_positions['gas'][2]
            if 2 in self.particle_positions['inlet_liquid']:
                self.particle_positions['inlet_liquid'][1] = self.particle_positions['inlet_liquid'][2]
            if 2 in self.particle_positions['inlet_gas']:
                self.particle_positions['inlet_gas'][1] = self.particle_positions['inlet_gas'][2]
            if 2 in self.particle_positions['inlet_grain']:
                self.particle_positions['inlet_grain'][1] = self.particle_positions['inlet_grain'][2]
            if 2 in self.particle_positions['grain']:
                self.particle_positions['grain'][1] = self.particle_positions['grain'][2]

    def create_or_get_emitter(self, emitter_name):
        if emitter_name not in bpy.data.objects:
            mesh = bpy.data.meshes.new(emitter_name)
            obj = bpy.data.objects.new(emitter_name, mesh)
            bpy.context.collection.objects.link(obj)
        else:
            obj = bpy.data.objects[emitter_name]
        return obj

    def create_emitters(self):
        if self.has_liquid:
            self.create_or_get_emitter("DF.Particles.Liquid")
        if self.has_gas:
            self.create_or_get_emitter("DF.Particles.Gas")
        if self.has_inlet_liquid:
            self.create_or_get_emitter("DF.Particles.Inlet_Liquid")
        if self.has_inlet_gas:
            self.create_or_get_emitter("DF.Particles.Inlet_Gas")
        if self.has_inlet_grain:
            self.create_or_get_emitter("DF.Particles.Inlet_Grain")
        if self.has_grain:
            self.create_or_get_emitter("DF.Particles.Grain")
        self.clear_frame_change_handler() 

    def delete_previous_emitters(self):
        for emitter_name in ["DF.Particles.Liquid", "DF.Particles.Gas", "DF.Particles.Inlet_Liquid", "DF.Particles.Inlet_Gas", "DF.Particles.Inlet_Grain" ,"DF.Particles.Grain"]:
            if emitter_name in bpy.data.objects:
                bpy.data.objects.remove(bpy.data.objects[emitter_name], do_unlink=True)
        self.particle_positions = {'liquid': {}, 'gas': {}, 'inlet_liquid': {}, 'inlet_gas': {}, 'inlet_grain': {} ,'grain': {}}
        for block in bpy.data.meshes:
            if block.users == 0:
                bpy.data.meshes.remove(block)
    
    def update_emitter_from_preloaded_data(self, emitter_name, frame_number):
        emitter_obj = self.create_or_get_emitter(emitter_name)
        points = None
        radius = None
        if emitter_name == "DF.Particles.Liquid" and frame_number in self.particle_positions['liquid']:
            points = self.particle_positions['liquid'][frame_number]
        elif emitter_name == "DF.Particles.Gas" and frame_number in self.particle_positions['gas']:
            points = self.particle_positions['gas'][frame_number]
        elif emitter_name == "DF.Particles.Inlet_Liquid" and frame_number in self.particle_positions['inlet_liquid']:
            points = self.particle_positions['inlet_liquid'][frame_number]
        elif emitter_name == "DF.Particles.Inlet_Gas" and frame_number in self.particle_positions['inlet_gas']:
            points = self.particle_positions['inlet_gas'][frame_number]
        elif emitter_name == "DF.Particles.Inlet_Grain" and frame_number in self.particle_positions['inlet_grain']:
            points, radius = self.particle_positions['inlet_grain'][frame_number]
        elif emitter_name == "DF.Particles.Grain" and frame_number in self.particle_positions['grain']:
            points, radius = self.particle_positions['grain'][frame_number]
        else:
            return
        if bpy.context.object and bpy.context.object.mode != 'OBJECT':
            bpy.ops.object.mode_set(mode='OBJECT')
        update_mesh_as_pointcloud(emitter_obj.data, points, radius)

    def update_particles_per_frame(self, scene, depsgraph):
        frame_number = scene.frame_current
        start_frame = scene.compute_mesh_properties.import_particles_start_frame
        end_frame = scene.compute_mesh_properties.import_particles_end_frame
        liquid_particles_exist = "DF.Particles.Liquid" in bpy.data.objects
        gas_particles_exist = "DF.Particles.Gas" in bpy.data.objects
        inlet_liquid_particles_exist = "DF.Particles.Inlet_Liquid" in bpy.data.objects
        inlet_gas_particles_exist = "DF.Particles.Inlet_Gas" in bpy.data.objects
        inlet_grain_particles_exist = "DF.Particles.Inlet_Grain" in bpy.data.objects
        grain_particles_exist = "DF.Particles.Grain" in bpy.data.objects
        if not liquid_particles_exist and not gas_particles_exist and not inlet_liquid_particles_exist and not inlet_gas_particles_exist and not inlet_grain_particles_exist and not grain_particles_exist:
            self.clear_frame_change_handler()
            print("Both emitters deleted. Stopping particle updates.")
            return
        if self.has_liquid and liquid_particles_exist and start_frame <= frame_number <= end_frame:
            self.update_emitter_from_preloaded_data("DF.Particles.Liquid", frame_number)
        if self.has_gas and gas_particles_exist and start_frame <= frame_number <= end_frame:
            self.update_emitter_from_preloaded_data("DF.Particles.Gas", frame_number)
        if self.has_inlet_liquid and inlet_liquid_particles_exist and start_frame <= frame_number <= end_frame:
            self.update_emitter_from_preloaded_data("DF.Particles.Inlet_Liquid", frame_number)
        if self.has_inlet_gas and inlet_gas_particles_exist and start_frame <= frame_number <= end_frame:
            self.update_emitter_from_preloaded_data("DF.Particles.Inlet_Gas", frame_number)
        if self.has_inlet_grain and inlet_grain_particles_exist and start_frame <= frame_number <= end_frame:
            self.update_emitter_from_preloaded_data("DF.Particles.Inlet_Grain", frame_number)
        if self.has_grain and grain_particles_exist and start_frame <= frame_number <= end_frame:
            self.update_emitter_from_preloaded_data("DF.Particles.Grain", frame_number)
      

    def setup_frame_change_handler(self):
        if self.update_particles_per_frame not in bpy.app.handlers.frame_change_pre:
            bpy.app.handlers.frame_change_pre.append(self.update_particles_per_frame)

    def clear_frame_change_handler(self):
        if self.update_particles_per_frame in bpy.app.handlers.frame_change_pre:
            bpy.app.handlers.frame_change_pre.remove(self.update_particles_per_frame)
            print("Frame change handler cleared.")

    def execute(self, context):
        blender_file_path = bpy.data.filepath
        blender_file_dir = os.path.dirname(blender_file_path)
        blender_file_name = os.path.splitext(os.path.basename(blender_file_path))[0]
        cache_folder_name = f"{blender_file_name}_cache"
        cache_folder_path = os.path.join(blender_file_dir, cache_folder_name)
        sim_output_name = f"{blender_file_name}_output"
        sim_output_directory = os.path.join(cache_folder_path, sim_output_name)
        self.delete_previous_emitters()
        self.preload_vtk_data(sim_output_directory)
        if not self.has_liquid and not self.has_gas and not self.has_inlet_liquid and not self.has_inlet_gas and not self.has_grain:
            self.clear_frame_change_handler()
            return {'CANCELLED'}
        self.create_emitters()
        self.setup_frame_change_handler()
        if self.particle_positions['liquid'] or self.particle_positions['gas'] or self.particle_positions['inlet_liquid'] or self.particle_positions['inlet_gas'] or self.particle_positions['grain']:
            bpy.context.scene.frame_set(context.scene.compute_mesh_properties.import_particles_start_frame)

        return {'FINISHED'}

def register():
    bpy.utils.register_class(DORIFLOW_OT_ImportAllParticles)

def unregister():
    bpy.utils.unregister_class(DORIFLOW_OT_ImportAllParticles)

if __name__ == "__main__":
    register()

