# Doriflow Engine - Fluid Simulation for Blender 3D
# Copyright (C) 2024 Doriflow Team
# This software is licensed under the Creative Commons Attribution-NonCommercial 4.0 International License (CC BY-NC 4.0).

# You are free to:
# -Share: Copy and redistribute the material in any medium or format.
# -Adapt: Remix, transform, and build upon the material.

# UNDER THE FOLOWING TERMS:
# -Attribution: You must give appropriate credit, provide a link to the license, and indicate if changes were made.
# -Appropriate credit should include the following:
#   -The original author's name: Doriflow Team
#   -A link to the original source (if applicable).
#   -A link to the full license: https://creativecommons.org/licenses/by-nc/4.0/.
#   -A clear indication of any changes made, such as: "This material has been modified."
# NonCommercial: You may not use the material for commercial purposes.
# Disclaimer:
# -This simulation engine is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. In no event shall the authors be liable for any claim, damages, or other liability arising from the use of this software.

# For more details, refer to the full license text at:
# https://creativecommons.org/licenses/by-nc/4.0/.

#-----------------------------------------------------------------------------------------------------------------------#
import json
import bpy
from bpy.props import (BoolProperty, IntProperty, FloatProperty, PointerProperty)
from bpy.types import PropertyGroup


def reset_cache(self, context):
    self.use_cache = False


class ComputeMeshProperties(PropertyGroup):
    is_import_running: BoolProperty(default=False)
    num_total_files: IntProperty(default=0)
    num_processed_files: IntProperty(default=0)
    is_compute_mesh_running: BoolProperty(default=False)
    is_compute_mesh_paused: BoolProperty(default=False)
    is_compute_mesh_stopped: BoolProperty(default=False)
    is_initialized: BoolProperty(default=False)
    is_validating: BoolProperty(default=False)
    std_dev: FloatProperty(
    name="Smoothing Factor",
    description="Controls the degree of mesh smoothing. Higher values result in a smoother mesh but may reduce mesh volume.",
    default=1,
    min=0.0001,
    max=100,
    update=reset_cache
)

    isosurface_level: FloatProperty(
        name="Mesh Detail Level",
        description="Determines the level of detail in the mesh reconstructed from fluid particles. Lower values capture more details but result in larger file sizes. Setting a value too low might produce the mesh wrap around the domain. Higher value will capture less details and curvature of the mesh but faster",
        default=0.4,
        min=0.0001,
        max=100,
        update=reset_cache
    )

    voxel_over_grid_factor: FloatProperty(
        name="Resolution Factor",
        description="Sets the mesh resolution. Lower values provide higher resolution, more mesh reconstruction around smaller cluster of particles. This parameter is very sensitive to changes; a recommended setting is 1 or 0.9. For highly splashing scene, usuing a value of 0.5 or even 0.3 help capturing more droplets at a higher cost resources",
        default=0.95,
        min=0.0001,
        max=100,
        update=reset_cache
    )

    use_cache: BoolProperty(name="Use Cache", default=True, description="If unchecked, the next run will erase all cached data and run from the start. If checked, the next run will continue from the last run")
    fluid_particle_percentage: bpy.props.FloatProperty(
        name="Fluid Particle Percentage",
        default=100,
        min=5,
        max=100,
        description="Control the mesh quality through varying the number of fluid particles used in the mesh reconstruction.",
        update=lambda self, context: self.update_fluid_particle_percentage(context)
    )
    import_particles_start_frame: IntProperty(
        name="Start Frame",
        description="The frame to start the mesh reconstruction process",
        default=1,
        min=1,
        max=100000
    )
    import_particles_end_frame: IntProperty(
        name="End Frame",
        description="The frame to end the mesh reconstruction process",
        default=250,
        min=1,
        max=100000
    )

    def update_fluid_particle_percentage(self, context):
        mesh_operator = bpy.ops.doriflow.mesh_geometry_nodes
        if mesh_operator.poll():
            mesh_operator()
    def reset(self):
        self.is_import_running = False
        self.is_initialized = False
        self.is_validating = False
        self.is_compute_mesh_paused = False
        self.is_compute_mesh_stopped = False
        self.is_compute_mesh_running = False

    def load_post(self):
        self.reset()
        
def register():
    bpy.utils.register_class(ComputeMeshProperties)
    bpy.types.Scene.compute_mesh_properties = bpy.props.PointerProperty(type=ComputeMeshProperties)


def unregister():
    bpy.utils.unregister_class(ComputeMeshProperties)
    del bpy.types.Scene.compute_mesh_properties
