# Doriflow Engine - Fluid Simulation for Blender 3D
# Copyright (C) 2024 Doriflow Team
# This software is licensed under the Creative Commons Attribution-NonCommercial 4.0 International License (CC BY-NC 4.0).

# You are free to:
# -Share: Copy and redistribute the material in any medium or format.
# -Adapt: Remix, transform, and build upon the material.

# UNDER THE FOLOWING TERMS:
# -Attribution: You must give appropriate credit, provide a link to the license, and indicate if changes were made.
# -Appropriate credit should include the following:
#   -The original author's name: Doriflow Team
#   -A link to the original source (if applicable).
#   -A link to the full license: https://creativecommons.org/licenses/by-nc/4.0/.
#   -A clear indication of any changes made, such as: "This material has been modified."
# NonCommercial: You may not use the material for commercial purposes.
# Disclaimer:
# -This simulation engine is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. In no event shall the authors be liable for any claim, damages, or other liability arising from the use of this software.

# For more details, refer to the full license text at:
# https://creativecommons.org/licenses/by-nc/4.0/.

#-----------------------------------------------------------------------------------------------------------------------#
import json
import bpy
from bpy.props import (FloatVectorProperty, FloatProperty, IntProperty, BoolProperty,EnumProperty, StringProperty)
from bpy.types import PropertyGroup

class DoriflowDomainProperties(PropertyGroup):
    particle_radius: FloatProperty(
        name="Particle Radius", 
        default=0.01, 
        min=0.001, 
        max=0.1
    )

    timestep_per_frame: IntProperty(
        name="Simulation Speed", 
        default=1, 
        min=1, 
        max=100, 
        description="Number of simulation steps per Blender's frame"
    )

    fluid_density: FloatProperty(
        name="Density", 
        default=1000, 
        min=1,  
        description="Density of the fluid"
    )

    gravitation: FloatVectorProperty(
        name="Gravity",
        description="Gravitation of the domain",
        default=(0.0, 0.0, -9.81),
        precision=2
    )

    time_step_size: FloatProperty(
        name="Time Step (ms)", 
        default=0.5,
        min=0, 
        precision=3,
        description=(
            "Simulation time step size in millisecond. Large value (e.g. 1 or 2) leads to faster "
            "simulation but may cause instability. Use small timestep (e.g. 0.01) "
            "for complex scenes or micrometer size physics such as droplet collisions"
        )
    )

    stiffness: FloatProperty(
        name="Stiffness", 
        default=50000, 
        min=10, 
        precision=0, 
        description=(
            "Defines the compressibility/splashing of the fluid and controls how much pressure "
            "changes with density. High stiffness values make fluids less compressible,"
            "causing higher pressure changes and more splashing effects. Suitable for simulating "
            "incompressible fluids like water, but requires smaller time steps, increasing "
            "computational cost and stability concerns. Low stiffness values allow for larger "
            "time steps and are ideal for compressible fluids, ideal for stable flow behaviours"
        )
    )

    exponent: IntProperty(
        name="Polytropic exponent", 
        default=2, 
        min=0, 
        description=(
            "Defines the nonlinearity of the pressure-density relationship and is often "
            "related to the specific properties of the fluid. High value can make the fluid "
            "more resistant to density changes, enhancing the representation of incompressible fluids,"
            "but potentially causing numerical instability, and requiring careful parameter tuning. "
            "Low exponent values are more appropriate for compressible fluids, enabling larger density "
            "changes with smaller pressure variations, but may fail to accurately represent "
            "incompressible fluids and can lead to excessive compressibility."
        )
    )

    resolution: IntProperty(
        name="Resolution", 
        default=50, 
        min=0,
        description="Number of grids along the longest dimension, with other dimensions scaled proportionally"
    )

    BCs: EnumProperty(
        name="Boundary Condition",
        items=[
            ('0', "No slip", "No-slip boundary condition : Fluid particles move with the boundary."),
            ('1', "Adhesion", "Adhesive boundary condition : Fluid particles stick to the boundary."),
        ],
        default='0',
    )

    viscosity: FloatProperty(
        name="Viscosity", 
        default=0.01, 
        min=0, 
        precision=7,
        description="Viscosity of the fluid. Default value is set at 0.05 for stability. Set higher value at around 0.1 for a more viscous fluids like honey, and lower value at around 0.01 for less viscous fluids"
    )

    surface_tension: FloatProperty(
        name="Surface Tension", 
        default=1,
        min=0,
        precision=7, 
        description="Surface tension of the fluid. Default value is set at water's surface tension"
    )

    liquid_drag_coefficient: FloatProperty(
        name="Liquid Drag Coefficient",
        default=0.1,
        min=0,
        precision=4,
        description="Drag coefficient of liquid phase"
    )
    export_fluid_velocity : BoolProperty(
        name="Export Fluid Velocity", 
        default=True,
        description=(
            "Enable to export fluid velocity data,which is compulsory for Whitewater computation, "
            "and to also visualize the velocity plots in Paraview. Disable for faster output writing if not required"
        )
    )

    export_rigid_pressure_force : BoolProperty(
        name="Export Rigid Pressure Force", 
        default=False, 
        description=(
            "Enable to export rigid pressure force data, and to visualize pressure forces on rigid bodies "
            "from fluid particles in Paraview. Useful for debugging pressure effects on rigid bodies"
        )
    )

    export_rigid_viscous_force : BoolProperty(
        name="Export Rigid Viscosity Force", 
        default=False, 
        description=(
            "Enable to export rigid viscous force data, and to visualize viscous forces on rigid bodies "
            "from fluid particles in Paraview. Useful for debugging viscous effects on rigid bodies"
        )
    )

    boundary_collision_factor: FloatProperty(
        name="Rigid Boundary Collision Factor", 
        default=0,
        description=(
            "Controls the collision intensity between rigid particles and domain boundaries. "
            "Increase to amplify the reaction forces on rigid bodies from domain boundaries (bouncing off effect)"
        )
    )

    rotation_factor: FloatProperty(
        name="Rotation Factor", 
        default=10, 
        description=(
            "Determines the intensity of the rotation matrix applied to rigid bodies, influencing "
            "the strength of their rotation. Particularly useful in low-resolution cases to achieve desired rotational effects"
        )
    )
    rigid_rigid_overlap_threshold: FloatProperty(name="Overlap Threshold", 
                                     default=10, min=1, precision=1,
                                     description="Factor to determine the minimum distance between two rigid bodies to avoid overlap."
                                     " Increase to prevent rigid bodies from overlapping. Decrease to prevent overshooting of rigid bodies.")
    
    velocity_stddev_multiplier: FloatProperty(
        name="Velocity Stddev Multiplier", 
        default=100, 
        min=0, 
        description=(
            "Multiplier to adjust the standard deviation of the velocity distribution. "
            "Increase to amplify the velocity variation of fluid particles, causing more turbulent flow"
        )
    )
    
    max_velocity_change_factor: FloatProperty(
        name="Max Velocity Change Factor", 
        default=10000, 
        min=0, 
        description=(
            "Factor to control the maximum velocity change of fluid particles. "
            "Increase to amplify the velocity change, causing more turbulent flow"
        )
    )
    
    advanced_variables: BoolProperty(
        name="Advanced variables of Liquid phase",
        description="Show advanced simulation parameters of Liquid Solver",
        default=False
    )
    viscosity_model: EnumProperty(
            name="Viscosity Model",
            items=[
                ("0", "Newtonian", "Newtonian viscosity model"),
                ("1", "Power Law", "Power Law viscosity model suits simulating non-Newtonian fluids like ketchup, shampoo, polymer solutions, or molten plastics."),
                ("2", "Carreau", "Carreau viscosity model suits simulating fluids with shear-thinning properties, such as blood, polymer melts, honey, or yogurt."),
                ("3", "Bingham", "Bingham viscosity model suits simulating fluids with yield stress, such as mud, toothpaste, mayonnaise, cement or slurries."),
                ("4", "Herschel-Bulkley", "Herschel-Bulkley viscosity model suits simulating fluids with yield stress, such as chocolate, peanut butter, or tomato paste"),
                # ("5", "Thixotropic", "Thixotropic viscosity model suits simulating fluids with time-dependent viscosity, such as paint, ink, or drilling mud."),
            ],
            default=0
            )
    # Power Law
    reference_shear_rate: FloatProperty(
        name="Reference Shear Rate",
        default=10.0,
        min=0.0,
        description="Reference shear rate for the Power Law model"
    )
    flow_behavior_index: FloatProperty(
        name="Flow Behavior Index",
        default=1.0,
        min=0.0,
        description="Flow behavior index for the Power Law model"
    )
    power_law_viscosity: FloatProperty(
        name="Power Law Viscosity",
        default=1.0,
        min=0.0,
        description="Viscosity for the Power Law model"
    )
    # Carreau
    carreau_mu_0: FloatProperty(
        name="Zero Shear Viscosity",
        default=10.0,
        min=0.0,
        description="Viscosity at zero shear rate for the Carreau model"
    )
    carreau_mu_inf: FloatProperty(
        name="Infinite Shear Viscosity",
        default=1.0,
        min=0.0,
        description="Viscosity at infinite shear rate for the Carreau model"
    )
    carreau_lambda: FloatProperty(
        name="Relaxation Time Constant",
        default=0.1,
        min=0.0,
        description="Relaxation time constant for the Carreau model"
    )
    carreau_flow_behavior_index: FloatProperty(
        name="Flow Behavior Index",
        default=0.5,
        min=0.0,
        description="Flow behavior index for the Carreau model"
    )
    # Bingham
    bingham_yield_stress: FloatProperty(
        name="Yield Stress",
        default=5.0,
        min=0.0,
        description="Yield stress for the Bingham model"
    )
    bingham_viscosity: FloatProperty(
        name="Base Viscosity",
        default=3.0,
        min=0.0,
        description="Base viscosity for the Bingham model"
    )
    # Herschel-Bulkley
    hb_yield_stress: FloatProperty(
        name="Yield Stress",
        default=5.0,
        min=0.0,
        description="Yield stress for the Herschel-Bulkley model"
    )
    hb_consistency_index: FloatProperty(
        name="Consistency Index (k)",
        default=0.2,
        min=0.0,
        description="Consistency index for the Herschel-Bulkley model"
    )
    hb_flow_behavior_index: FloatProperty(
        name="Flow Behavior Index (n)",
        default=0.5,
        min=0.0,
        description="Flow behavior index for the Herschel-Bulkley model"
    )
    thixotropy_a: FloatProperty(
        name="Thixotropy A",
        default=0.1,
        min=0.0,
        description="Thixotropy A for the Thixotropic model"
    )
    thixotropy_b: FloatProperty(
        name="Thixotropy B",
        default=0.1,
        min=0.0,
        description="Thixotropy B for the Thixotropic model"
    )
    thixotropy_mu_0: FloatProperty(
        name="Zero Shear Viscosity",
        default=10.0,
        min=0.0,
        description="Viscosity at zero shear rate for the Thixotropic model"
    )
    thixotropy_mu_inf: FloatProperty(
        name="Infinite Shear Viscosity",
        default=1.0,
        min=0.0,
        description="Viscosity at infinite shear rate for the Thixotropic model"
    )
    particles_delete_at_boundary: BoolProperty(
        name="Delete Particles at Boundary",
        default=False,
        description="Enable to delete particles that at the domain boundary"
    )
# Gas Properties
    R_gas: FloatProperty(
        name="Gas Constant",
        default=287.0,
        min=0.0,
        description="Controls how fast gas expands or contracts with pressure or temperature changes. Higher values make the gas expand faster."
    )

    gas_density: FloatProperty(
        name="Gas Density",
        default=1.0,
        min=0.0,
        description="Defines how heavy the gas is per unit volume. Higher values make the gas denser, while lower values make it lighter."
    )

    gas_viscosity: FloatProperty(
        name="Gas Viscosity",
        default=0.01,
        min=0.0,
        description="Determines how 'thick' or 'sticky' the gas is when it moves. Higher values make the gas flow slower and smoother."
    )

    ambient_pressure: FloatProperty(
        name="Ambient Pressure",
        default=101325.0,
        min=0.0,
        description="Represents the surrounding air pressure. Higher values compress the gas more, while lower values let it expand more freely."
    )

    pressure_scale: FloatProperty(
        name="Pressure Scale",
        default=0.001,
        min=0.0,
        description="Adjusts how much pressure affects gas behavior. Higher values make pressure changes have a stronger effect in gas particles motion"
    )

    drag_coefficient: FloatProperty(
        name="Drag Coefficient",
        default=1,
        min=0.0,
        description="Controls how much the gas slows down when it moves. Higher values make the gas slow down faster"
    )
    buoyancy_coefficient: FloatProperty(
        name="Buoyancy Coefficient",
        default=1.0,
        min=0.0,
        description="Controls how strongly the gas rises or falls due to differences in temperature. Higher values make the gas rise faster"
    )

    diffusion_coefficient: FloatProperty(
        name="Diffusion Coefficient",
        default=0.01,
        min=0.0,
        description="Determines how quickly gas spreads and mixes with its surroundings. Higher values make gas spread more evenly"
    )

    vorticity_epsilon: FloatProperty(
        name="Vorticity Epsilon",
        default=0.5,
        min=0.0,
        description="Affects how much swirling or turbulence happens in the gas flow. Higher values increase swirling effects"
    )

    smagorinsky_coefficient: FloatProperty(
        name="Smagorinsky Coefficient",
        default=0.16,
        min=0.0,
        description="Controls small-scale turbulence smoothing. Higher values reduce fine details in the gas flow"
    )

    delta_T: FloatProperty(
        name="Temperature Difference",
        default=1.0,
        description="Represents the temperature difference between the gas and its surroundings. Higher postive values make gas rise faster. Lower negative values make it sink faster"
    )

    dissipation_factor: FloatProperty(
        name="Dissipation Factor",
        default=0.99,
        min=0.0,
        description="Determines how quickly gas particle density reduce overtime. Lower values make gas particles disappear quicker"
    )

    particle_deactivation_threshold: FloatProperty(
        name="Particle Deactivation Threshold",
        default=0.0,
        min=0.0,
        max=100.0,
        description="Percentage of the bulk density below which the particle will be deleted. Lower values remove the particles far away from the bulk. Higher values remove the particles closer to the bulk"
    )

    advanced_variables_gas: BoolProperty(
        name="Advanced Variables of Gas Phase",
        description="Enable this to access additional settings for fine-tuning the gas simulation.",
        default=False
    )

    particles_delete_at_boundary_gas: BoolProperty(
        name="Delete Particles at Boundary",
        default=False,
        description="If enabled, gas particles touching the domain boundary will be removed, preventing lingering at the edges."
    )

#DEM
    grain_density: FloatProperty(
        name="Density",
        default=1000,
        min=0,
        max=10000,
        precision=2,
        description="Density of the grains"
    )

    radius_std: FloatProperty(
        name="Radius Variation",
        default=10,
        min=0,
        precision=2,
        description="Percentage Variation in radius of the grains"
    )
    
    dem_stiffness: FloatProperty(
        name="DEM Stiffness",
        default=50,
        min=0,
        precision=3,
        description=(
            "Controls the repulsion force between grains and affects the bulk volume of particles. "
            "Increase this value if grains penetrate through solid boundaries. "
            "Decrease this value for a more stable and smoother granular flow behavior during settling."
        )
    )
    
    dem_damping_coefficient: FloatProperty(
        name="Damping Coefficient",
        default=0.01,
        min=0,
        precision=3,
        description="Damping coefficient of the grains motion"
    )
    
    dem_friction_coefficient: FloatProperty(
        name="Friction Coefficient",
        default=0.1,
        min=0,
        max=1,
        precision=2,
        description="Friction coefficient between granular particles"
    )
    
    dem_drag_coefficient: FloatProperty(
        name="Drag Coefficient",
        default=0.1,
        min=0,
        precision=2,
        description="Drag coefficient of the grains"
    )
    
    dem_boundary_friction_coefficient: FloatProperty(
        name="Boundary Friction Coefficient",
        default=0.1,
        min=0,
        max=1,
        precision=2,
        description="Friction coefficient of the grains with the boundary"
    )
    
    dem_boundary_bouncing_coefficient: FloatProperty(
        name="Boundary Bouncing Coefficient",
        default=0.1,
        min=0,
        precision=2,
        description="Bouncing coefficient of the grains with the boundary"
    )
    
    bond_modulus: FloatProperty(
        name="Bond Modulus",
        default=1e3,
        min=0,
        precision=2,
        description="Modulus of the bonds between grains"
    )
    
    bond_damping_coefficient: FloatProperty(
        name="Bond Damping Coefficient",
        default=0.01,
        min=0,
        precision=3,
        description="Damping coefficient of the bonds between grains"
    )
    
    bond_tensile_strength: FloatProperty(
        name="Bond Tensile Strength",
        default=0,
        min=0,
        precision=2,
        description="Tensile strength of the bonds between grains"
    )
    
    bond_shear_strength: FloatProperty(
        name="Bond Shear Strength",
        default=0,
        min=0,
        precision=2,
        description="Shear strength of the bonds between grains"
    )
    
    dem_max_neighbours: IntProperty(
        name="Max Neighbor",
        default=20,
        min=0,
        description="Maximum number of neighbors for each grain"
    )
    advanced_variables_grain: BoolProperty(
        name="Advanced Variables of Grain Phase",
        description="Enable this to access additional settings for fine-tuning the grain simulation.",
        default=False
    )
    
    rigid_bouncing_coefficient: FloatProperty(
        name="Rigid Bouncing Coefficient",
        default=0.1,
        min=0,
        precision=2,
        description="Bouncing coefficient of the rigid body"
    )
    
    rigid_damping_coefficient: FloatProperty(
        name="Rigid Damping Coefficient",
        default=0.1,
        min=0,
        precision=2,
        description="Damping coefficient of the grigid body"
    )
    
    rigid_friction_coefficient: FloatProperty(
        name="Rigid Friction Coefficient",
        default=0.1,
        min=0,
        max=1,
        precision=2,
        description="Friction coefficient of the rigid body"
    )
    
    advanced_variables_rigid_body: BoolProperty(
        name="Advanced Variables of Rigid Body",
        description="Enable this to access additional settings for fine-tuning the rigid body simulation.",
        default=False
    )
def register():
    bpy.utils.register_class(DoriflowDomainProperties)


def unregister():
    bpy.utils.unregister_class(DoriflowDomainProperties)

