# Doriflow Engine - Fluid Simulation for Blender 3D
# Copyright (C) 2024 Doriflow Team
# This software is licensed under the Creative Commons Attribution-NonCommercial 4.0 International License (CC BY-NC 4.0).

# You are free to:
# -Share: Copy and redistribute the material in any medium or format.
# -Adapt: Remix, transform, and build upon the material.

# UNDER THE FOLOWING TERMS:
# -Attribution: You must give appropriate credit, provide a link to the license, and indicate if changes were made.
# -Appropriate credit should include the following:
#   -The original author's name: Doriflow Team
#   -A link to the original source (if applicable).
#   -A link to the full license: https://creativecommons.org/licenses/by-nc/4.0/.
#   -A clear indication of any changes made, such as: "This material has been modified."
# NonCommercial: You may not use the material for commercial purposes.
# Disclaimer:
# -This simulation engine is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. In no event shall the authors be liable for any claim, damages, or other liability arising from the use of this software.

# For more details, refer to the full license text at:
# https://creativecommons.org/licenses/by-nc/4.0/.

#-----------------------------------------------------------------------------------------------------------------------#
import json
import bpy

def draw_import_mesh_operator(context, layout):
    global_props = context.scene.doriflow
    if global_props is None:
        return

    compute_mesh_props = global_props.compute_mesh
    if not compute_mesh_props.is_compute_mesh_running:
        operator_text = "Compute"
    elif not compute_mesh_props.is_initialized:
        operator_text = "Computing Mesh in progress... initializing"
    elif compute_mesh_props.is_compute_mesh_paused:
        frame_str = str(compute_mesh_props.num_processed_files)
        operator_text = "Resume Computation (from frame " + frame_str + ")"
    elif compute_mesh_props.is_compute_mesh_stopped:
        operator_text = "Cancelling... "
    else:
        num_files = compute_mesh_props.num_total_files
        progress_string = str(compute_mesh_props.num_processed_files) + " / " + str(num_files)
        pct_string = str(round((compute_mesh_props.num_processed_files / num_files) * 100, 1)) + "%"
        compute_mesh_progress_string = pct_string + "  (" + progress_string + ")"
        operator_text = "Computing Mesh in progress...   " + compute_mesh_progress_string
    column = layout.column(align=True)
    if compute_mesh_props.is_compute_mesh_paused:
        split = column.split(factor=0.75, align=True)
        column_left = split.column(align=True)
        column_left.operator("doriflow.resume_compute_mesh", text=operator_text)
        column_right = split.column(align=True)
        column_right.alert = True
        column_right.operator("doriflow.stop_compute_mesh", text="Stop")
    else:
        column.operator("doriflow.compute_mesh", text=operator_text, icon = 'VIEW_ORTHO')
        if compute_mesh_props.is_compute_mesh_running:
            column.operator("doriflow.pause_compute_mesh", text="Pause")


class DORIFLOW_PT_ComputeMeshPanel(bpy.types.Panel):
    bl_label = "[Doriflow] -3- Mesh"
    bl_space_type = 'PROPERTIES'
    bl_region_type = 'WINDOW'
    bl_category = 'Doriflow'
    bl_context = "physics"
    bl_order = 20
    bl_options = {'DEFAULT_CLOSED'}

    @classmethod
    def poll(cls, context):
        return context.object is not None and context.object.doriflow.object_type == 'TYPE_DOMAIN' and context.object.doriflow.is_active

    def draw(self, context):
        layout = self.layout
        box2 = layout.box()
        scene = context.scene
        compute_mesh_props = scene.compute_mesh_properties
        box2.label(text="Import Particles Frame Range:")
        split = box2.split(factor=0.5)
        col = split.column()
        col.prop(compute_mesh_props, "import_particles_start_frame", text="Start Frame")
        col = split.column()
        col.prop(compute_mesh_props, "import_particles_end_frame", text="End Frame")
        box2.operator("doriflow.import_all_particles", text=" Import All Particles")
        box2.operator("doriflow.mesh_geometry_nodes", text="Fluid Particles to Mesh")
        box2.operator("doriflow.grain_geometry_nodes", text="Granular Particles to Grains")

def register():
    bpy.utils.register_class(DORIFLOW_PT_ComputeMeshPanel)

def unregister():
    bpy.utils.unregister_class(DORIFLOW_PT_ComputeMeshPanel)

if __name__ == "__main__":
    register()
