# Doriflow Engine - Fluid Simulation for Blender 3D
# Copyright (C) 2024 Doriflow Team
# This software is licensed under the Creative Commons Attribution-NonCommercial 4.0 International License (CC BY-NC 4.0).

# You are free to:
# -Share: Copy and redistribute the material in any medium or format.
# -Adapt: Remix, transform, and build upon the material.

# UNDER THE FOLOWING TERMS:
# -Attribution: You must give appropriate credit, provide a link to the license, and indicate if changes were made.
# -Appropriate credit should include the following:
#   -The original author's name: Doriflow Team
#   -A link to the original source (if applicable).
#   -A link to the full license: https://creativecommons.org/licenses/by-nc/4.0/.
#   -A clear indication of any changes made, such as: "This material has been modified."
# NonCommercial: You may not use the material for commercial purposes.
# Disclaimer:
# -This simulation engine is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. In no event shall the authors be liable for any claim, damages, or other liability arising from the use of this software.

# For more details, refer to the full license text at:
# https://creativecommons.org/licenses/by-nc/4.0/.

#-----------------------------------------------------------------------------------------------------------------------#
import json
import bpy

class RenderAspectRatioProperties(bpy.types.PropertyGroup):
    aspect_ratio: bpy.props.EnumProperty(
        name="Aspect Ratio",
        description="Select a preset aspect ratio",
        items=[
            ("16:9", "16:9", "Standard HD"),
            ("1:1", "1:1", "Square"),
            ("4:3", "4:3", "Standard Definition"),
            ("2.35:1", "2.35:1", "CinemaScope")
        ],
        update=lambda self, context: update_aspect_ratio(self, context)
    )

def update_aspect_ratio(self, context):
    aspect_ratios = {
        "16:9": (1920, 1080),
        "1:1": (1080, 1080),
        "4:3": (1440, 1080),
        "2.35:1": (1920, 819)
    }
    res_x, res_y = aspect_ratios[self.aspect_ratio]
    context.scene.render.resolution_x = res_x
    context.scene.render.resolution_y = res_y

class DORIFLOW_PT_RenderFinalSimulationPanel(bpy.types.Panel):
    bl_space_type = "PROPERTIES"
    bl_region_type = "WINDOW"
    bl_context = "physics"
    bl_category = "Doriflow"
    bl_label = "[Doriflow] -5- Render Doriflow Animation"
    bl_order = 60
    bl_options = {'DEFAULT_CLOSED'}

    @classmethod
    def poll(cls, context):
        return (context.object is not None and
                context.object.doriflow.object_type == 'TYPE_DOMAIN' and
                context.object.doriflow.is_active)

    def draw(self, context):
        layout = self.layout
        scene = context.scene
        box = layout.box()
        row = box.row()
        row.label(text="Aspect Ratio:") 
        row.prop(scene.render_aspect_ratio_properties, "aspect_ratio", text="")
        row = box.row()
        row.prop(scene.doriflow, "render_output_path", text="Output Path")
        row.operator("doriflow.select_render_output_path", text="", icon='FILE_FOLDER')  
        box.label(text="Render Frame Range:")
        row = box.row()
        row.prop(scene.doriflow, "render_start_frame", text="Start Frame")
        row.prop(scene.doriflow, "render_end_frame", text="End Frame")
        layout.operator("doriflow.custom_render_ww", text="Render Final Animation",  icon='RENDER_ANIMATION')

def register():
    bpy.utils.register_class(RenderAspectRatioProperties)
    bpy.types.Scene.render_aspect_ratio_properties = bpy.props.PointerProperty(type=RenderAspectRatioProperties)
    bpy.utils.register_class(DORIFLOW_PT_RenderFinalSimulationPanel)

def unregister():
    bpy.utils.unregister_class(DORIFLOW_PT_RenderFinalSimulationPanel)
    bpy.utils.unregister_class(RenderAspectRatioProperties)
    del bpy.types.Scene.render_aspect_ratio_properties

if __name__ == "__main__":
    register()
