# Doriflow Engine - Fluid Simulation for Blender 3D
# Copyright (C) 2024 Doriflow Team
# This software is licensed under the Creative Commons Attribution-NonCommercial 4.0 International License (CC BY-NC 4.0).

# You are free to:
# -Share: Copy and redistribute the material in any medium or format.
# -Adapt: Remix, transform, and build upon the material.

# UNDER THE FOLOWING TERMS:
# -Attribution: You must give appropriate credit, provide a link to the license, and indicate if changes were made.
# -Appropriate credit should include the following:
#   -The original author's name: Doriflow Team
#   -A link to the original source (if applicable).
#   -A link to the full license: https://creativecommons.org/licenses/by-nc/4.0/.
#   -A clear indication of any changes made, such as: "This material has been modified."
# NonCommercial: You may not use the material for commercial purposes.
# Disclaimer:
# -This simulation engine is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. In no event shall the authors be liable for any claim, damages, or other liability arising from the use of this software.

# For more details, refer to the full license text at:
# https://creativecommons.org/licenses/by-nc/4.0/.

#-----------------------------------------------------------------------------------------------------------------------#
import json
import bpy
from ..utils import format_list, calculate_bounding_box

def add_collapsible_properties():
    bpy.types.Object.show_liquid = bpy.props.BoolProperty(name="Show Liquid Variables", default=False)
    bpy.types.Object.show_gas = bpy.props.BoolProperty(name="Show Gas Variables", default=False)
    bpy.types.Object.show_grain = bpy.props.BoolProperty(name="Show Grain Variables", default=False)
    bpy.types.Object.show_rigid = bpy.props.BoolProperty(name="Show Rigid Variables", default=False)
    
class DORIFLOW_PT_DomainTypePanel(bpy.types.Panel):
    bl_space_type = "PROPERTIES"
    bl_region_type = "WINDOW"
    bl_context = "physics"
    bl_category = "Doriflow"
    bl_label = "Solver Initialization"
    bl_parent_id = "OBJECT_PT_initialization_ui"

    @classmethod
    def poll(cls, context):
        return context.object is not None and context.object.doriflow.object_type == 'TYPE_DOMAIN' and context.object.doriflow.is_active

    def draw(self, context):
        layout = self.layout
        obj_props = context.object.doriflow
        domain_props = obj_props.domain
        solver_box = layout.box()

        solver_properties = [
            ("Resolution", "resolution"),
            ("Time Step Size (ms)", "time_step_size"),
        ]
        for label, prop in solver_properties:
            row = solver_box.row()
            row.label(text=label)
            row.prop(domain_props, prop, text="")
        solver_box.prop(context.object, "show_liquid", text="Liquid Solver", icon='MOD_FLUIDSIM', emboss=True)
        if context.object.show_liquid:
            basic_properties = [
                ("Density", "fluid_density"),
                ("Viscosity", "viscosity"),
            ]
            for label, prop in basic_properties:
                row = solver_box.row()
                row.label(text=label)
                row.prop(domain_props, prop, text="")
        solver_box.prop(context.object, "show_gas", text="Gas Solver", icon='MOD_FLUID', emboss=True)

        if context.object.show_gas:
            row = solver_box.row()
            row.label(text="This feature is available in the Doriflow Full version.")

        solver_box.prop(context.object, "show_grain", text="Grain Solver", icon='POINTCLOUD_DATA', emboss=True)
        
        if context.object.show_grain:

            basic_grain_properties = [
                ("Density", "grain_density"),
            ]
            for label, prop in basic_grain_properties:
                row = solver_box.row()
                row.label(text=label)
                row.prop(domain_props, prop, text="")
        solver_box.prop(context.object, "show_rigid", text="Rigid Body Solver", icon='POINTCLOUD_DATA', emboss=True)
        if context.object.show_rigid:
            basic_rigid_properties = [
                ("Rigid Bouncing Coefficient", "rigid_bouncing_coefficient"),
                ("Rigid Friction Coefficient", "rigid_friction_coefficient"),
            ]
            for label, prop in basic_rigid_properties:
                row = solver_box.row()
                row.label(text=label)
                row.prop(domain_props, prop, text="")
                

def register():
    bpy.utils.register_class(DORIFLOW_PT_DomainTypePanel)
    add_collapsible_properties()

def unregister():
    bpy.utils.unregister_class(DORIFLOW_PT_DomainTypePanel)
    del bpy.types.Object.show_liquid
    del bpy.types.Object.show_gas
    del bpy.types.Object.show_grain
  