# Doriflow Engine - Fluid Simulation for Blender 3D
# Copyright (C) 2024 Doriflow Team
# This software is licensed under the Creative Commons Attribution-NonCommercial 4.0 International License (CC BY-NC 4.0).

# You are free to:
# -Share: Copy and redistribute the material in any medium or format.
# -Adapt: Remix, transform, and build upon the material.

# UNDER THE FOLOWING TERMS:
# -Attribution: You must give appropriate credit, provide a link to the license, and indicate if changes were made.
# -Appropriate credit should include the following:
#   -The original author's name: Doriflow Team
#   -A link to the original source (if applicable).
#   -A link to the full license: https://creativecommons.org/licenses/by-nc/4.0/.
#   -A clear indication of any changes made, such as: "This material has been modified."
# NonCommercial: You may not use the material for commercial purposes.
# Disclaimer:
# -This simulation engine is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. In no event shall the authors be liable for any claim, damages, or other liability arising from the use of this software.

# For more details, refer to the full license text at:
# https://creativecommons.org/licenses/by-nc/4.0/.

#-----------------------------------------------------------------------------------------------------------------------#
import bpy
import json
import os
import numpy as np
from mathutils import Matrix, Vector, Euler
from bpy.types import Operator
from .import_all_particles_operators import DORIFLOW_OT_ImportAllParticles as df_import_particles


class DORIFLOW_OT_ImportMotionData(Operator):
    bl_idname = "doriflow.import_motion_data"
    bl_label = "Import Motion Data to Multiple Objects"
    bl_options = {'REGISTER'}
    bl_description = "Import motion data (translational and rotational matrix) to multiple objects from a simulation cache folder."
    def calculate_initial_center_of_mass(self, obj):
        mesh = obj.data
        verts = [v.co for v in mesh.vertices]
        center = sum(verts, Vector()) / len(verts)
        return center
    def import_transformation(self, obj, com, rotation_matrix, frame):
        obj.location = Vector(com)
        rot_mat = Matrix(rotation_matrix)
        obj.rotation_mode = 'QUATERNION'
        obj.rotation_quaternion = rot_mat.to_quaternion()
        obj.keyframe_insert(data_path="location", frame=frame)
        obj.keyframe_insert(data_path="rotation_quaternion", frame=frame)
    def create_object_id_to_blender_name_mapping(self,json_inputs_path):
        with open(json_inputs_path, 'r') as file:
            input_data = json.load(file)
        mapping = {}
        for body in input_data['RigidBodies']:
            object_id = body['objectId']
            blender_name = body['blenderName']
            mapping[object_id] = blender_name
        return mapping
    def remove_object_and_mesh(self, obj_name):
        obj = bpy.data.objects.get(obj_name)
        if obj:
            bpy.data.objects.remove(obj, do_unlink=True)
            print(f"Removed object: {obj_name}")
        mesh = bpy.data.meshes.get(obj_name)
        if mesh and mesh.users == 0:
            bpy.data.meshes.remove(mesh)
            print(f"Removed mesh: {obj_name}")
    def execute(self, context):
        bpy.context.scene.frame_set(1)
        blender_file_path = bpy.data.filepath
        if not blender_file_path:
            self.report({'ERROR'}, "Blender file is not saved. Please save the file before running the simulation.")
            return {'CANCELLED'}
        blender_file_dir = os.path.dirname(blender_file_path)
        blender_file_name = os.path.splitext(os.path.basename(blender_file_path))[0]
        cache_folder_name = f"{blender_file_name}_cache"
        cache_folder_path = os.path.join(blender_file_dir, cache_folder_name)
        sim_output_name = f"{blender_file_name}_output"
        sim_output_path = os.path.join(cache_folder_path, sim_output_name)
        json_motion_data_name ="motion_data"
        json_motion_data_path = os.path.join(sim_output_path, json_motion_data_name)
        json_inputs_name = f"{blender_file_name}_initial_condition.json"
        json_inputs_path = os.path.join(cache_folder_path, json_inputs_name)
        cleared = False#
        for obj in bpy.data.objects:
            if obj.name.startswith("DF.Particles") or obj.name.startswith("DF.Mesh"):
                self.remove_object_and_mesh(obj.name)
                cleared = True
        messages = []
        if cleared:
            messages.append(
                "Importing solid motion will clear out fluid particles and meshes. "
                "If this happens, please kindly import the fluid particles again."
            )
        id_to_blender_name_mapping = self.create_object_id_to_blender_name_mapping(json_inputs_path)
        if not os.path.exists( json_motion_data_path):
            self.report({'ERROR'}, f"Directory not found: { json_motion_data_path}")
            return {'CANCELLED'}
        
        json_motion_files = sorted(f for f in os.listdir( json_motion_data_path) if f.startswith('motion_data') and f.endswith('.json'))
        bpy.ops.object.select_all(action='DESELECT')
        for obj_id, blender_obj_name in self.create_object_id_to_blender_name_mapping(json_inputs_path).items():
            obj = bpy.data.objects.get(blender_obj_name)
            if obj and  obj.doriflow.obstacle.isFloating:
                context.view_layer.objects.active = obj
                obj.select_set(True)
                bpy.ops.object.transform_apply(location=False, rotation=True, scale=False)
                obj.select_set(False) 
                
        # Read frame 1 data and compute offset
        frame_1_motion_file = json_motion_files[0]
        frame_1_path = os.path.join(json_motion_data_path, frame_1_motion_file)

        with open(frame_1_path, 'r') as file:
            frame_1_data = json.load(file)

        object_offsets = {}  # Store offsets for each object

        for obj_id, motion_data in frame_1_data.items():
            blender_obj_name = id_to_blender_name_mapping.get(int(obj_id))
            obj = bpy.data.objects.get(blender_obj_name)
            if obj and obj.doriflow.object_type == 'TYPE_OBSTACLE' and obj.doriflow.obstacle.isFloating:
                com_solver = Vector(motion_data['center_of_mass'])
                com_blender = obj.location.copy()
                offset_com = com_blender - com_solver
                object_offsets[int(obj_id)] = offset_com
                
        for obj in bpy.data.objects:
            if obj.doriflow.object_type == 'TYPE_OBSTACLE' and obj.doriflow.obstacle.isFloating:
                obj.animation_data_clear()
                initial_location = obj.location.copy()
                initial_rotation = obj.rotation_euler.copy()
                bpy.context.scene.frame_set(1)
                obj.location = initial_location
                obj.rotation_euler = initial_rotation
                obj.keyframe_insert(data_path="location", frame=1)
                obj.keyframe_insert(data_path="rotation_quaternion", frame=1)
        for frame_number, json_file in enumerate(json_motion_files, start=2):
            file_path = os.path.join(json_motion_data_path, json_file)
            with open(file_path, 'r') as file:
                data = json.load(file)
                for obj_id, motion_data in data.items():
                    blender_obj_name = id_to_blender_name_mapping[int(obj_id)]
                    obj = bpy.data.objects.get(blender_obj_name)
                    if obj.doriflow.object_type == 'TYPE_OBSTACLE' and obj.doriflow.obstacle.isFloating:
                        com = motion_data['center_of_mass']
                        rotation_matrix = motion_data['rotation_matrix']
                        bpy.context.scene.frame_set(frame_number+1)
                        self.import_transformation(obj, com, rotation_matrix, frame_number)
        for obj in bpy.data.objects:
            if obj.doriflow.object_type == 'TYPE_DOMAIN':
                domain_object = obj
                bpy.context.view_layer.objects.active = domain_object
                domain_object.select_set(True)     
        messages.append("Motion data applied successfully.")          
        self.report({'INFO'}, " ".join(messages))
        return {'FINISHED'}

def register():
    bpy.utils.register_class(DORIFLOW_OT_ImportMotionData)

def unregister():
    bpy.utils.unregister_class(DORIFLOW_OT_ImportMotionData)

if __name__ == "__main__":
    register()
