# Doriflow Engine - Fluid Simulation for Blender 3D
# Copyright (C) 2024 Doriflow Team
# This software is licensed under the Creative Commons Attribution-NonCommercial 4.0 International License (CC BY-NC 4.0).

# You are free to:
# -Share: Copy and redistribute the material in any medium or format.
# -Adapt: Remix, transform, and build upon the material.

# UNDER THE FOLOWING TERMS:
# -Attribution: You must give appropriate credit, provide a link to the license, and indicate if changes were made.
# -Appropriate credit should include the following:
#   -The original author's name: Doriflow Team
#   -A link to the original source (if applicable).
#   -A link to the full license: https://creativecommons.org/licenses/by-nc/4.0/.
#   -A clear indication of any changes made, such as: "This material has been modified."
# NonCommercial: You may not use the material for commercial purposes.
# Disclaimer:
# -This simulation engine is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. In no event shall the authors be liable for any claim, damages, or other liability arising from the use of this software.

# For more details, refer to the full license text at:
# https://creativecommons.org/licenses/by-nc/4.0/.

#-----------------------------------------------------------------------------------------------------------------------#
import bpy
from bpy.types import Operator
from bpy.props import StringProperty
import json
from ..utils import format_list, calculate_bounding_box
from mathutils import Vector
import os
import numpy as np
from collections import defaultdict
import math
import sys


class CompactJSONEncoder(json.JSONEncoder):
    def iterencode(self, o, _one_shot=False):
        if isinstance(o, list):
            yield '[' + ', '.join(json.dumps(el, cls=CompactJSONEncoder) for el in o) + ']'
        else:
            yield from super().iterencode(o, _one_shot)


class DORIFLOW_OT_ExportData(Operator):
    bl_idname = "doriflow.export_data"
    bl_label = "Export DoriFlow Configuration"
    bl_options = {'REGISTER'}

    def execute(self, context):
        bpy.ops.object.select_all(action='DESELECT')
        for obj in bpy.data.objects:
            if obj.doriflow.object_type != 'TYPE_NONE':
                obj.select_set(True)
                bpy.context.view_layer.objects.active = obj
                bpy.ops.object.transform_apply(location=False, rotation=False, scale=False)
            else:
                obj.select_set(False)
        data = {
            "Domain": {},
            "FluidBlocks": [],
            "RigidBodies": [], 
            "KeyframedObstacles": [],
            "InletObjects": [],
            "OutletObjects": [],
            "GrainObjects": []
        }
        blender_file_path = bpy.data.filepath
        if not blender_file_path:
            self.report({'ERROR'}, "Blender file is not saved. Please save the file before running the export.")
            return {'CANCELLED'}
        blender_file_dir = os.path.dirname(blender_file_path)
        blender_file_name = os.path.splitext(os.path.basename(blender_file_path))[0]
        cache_folder_name = f"{blender_file_name}_cache"
        cache_folder_path = os.path.join(blender_file_dir, cache_folder_name)
        export_folder_name = f"{blender_file_name}_keyframed_motion"
        export_folder_path = os.path.join(cache_folder_path, export_folder_name)
        os.makedirs(cache_folder_path, exist_ok=True)
        json_file_name = f"{blender_file_name}_initial_condition.json"
        json_file_path = os.path.join(cache_folder_path, json_file_name)
        for obj in bpy.data.objects:
            if obj.doriflow.object_type == 'TYPE_DOMAIN':
                min_point, max_point = calculate_bounding_box(obj)
                translation_vector = [-coord for coord in min_point]
                obj_domain = obj.doriflow.domain
                data["Domain"]["geometryFile"] = obj.doriflow.obj_path
                data["Domain"]["operating_system"] = sys.platform
                data["Domain"]["domainStart"] = list(min_point)
                data["Domain"]["domainStart_1"] = list(min_point)
                data["Domain"]["domainStart_2"] = list(min_point)
                data["Domain"]["domainEnd"] = max_point
                data["Domain"]["translation_vector"] = list(translation_vector)
                data["Domain"]["timestep_per_frame"] = obj_domain.timestep_per_frame
                data["Domain"]["resolution"] = obj_domain.resolution
                data["Domain"]["gravitation"] = list(obj_domain.gravitation)
                data["Domain"]["timeStepSize"] = round(obj_domain.time_step_size/1000,8)
                data["Domain"]["stiffness"] = obj_domain.stiffness
                data["Domain"]["exponent"] = obj_domain.exponent
                data["Domain"]["BCs"] = obj_domain.BCs
                data["Domain"]["viscosity"] = round(obj_domain.viscosity,6)
                data["Domain"]["surface_tension"] = round(obj_domain.surface_tension,6)
                data["Domain"]["fluid_density_reference"] = 1000
                data["Domain"]["fluid_density"] = obj_domain.fluid_density
                data["Domain"]["liquid_drag_coefficient"] = obj_domain.liquid_drag_coefficient
                data["Domain"]["particles_delete_at_boundary"] = obj_domain.particles_delete_at_boundary
                data["Domain"]["boundary_collision_factor"] = obj_domain.boundary_collision_factor
                data["Domain"]["rotation_factor"] = obj_domain.rotation_factor
                data["Domain"]["rigid_rigid_overlap_threshold"] = obj_domain.rigid_rigid_overlap_threshold
                data["Domain"]["velocity_stddev_multiplier"] = obj_domain.velocity_stddev_multiplier
                data["Domain"]["max_velocity_change_factor"] = obj_domain.max_velocity_change_factor
                data["Domain"]["start_frame"] = bpy.context.scene.frame_start
                data["Domain"]["end_frame"] = bpy.context.scene.frame_end
                data["Domain"]["fps"] = bpy.context.scene.render.fps
                data["Domain"]["export_fluid_velocity"] = obj_domain.export_fluid_velocity
                data["Domain"]["export_rigid_pressure_force"] = obj_domain.export_rigid_pressure_force
                data["Domain"]["export_rigid_viscous_force"] = obj_domain.export_rigid_viscous_force
                data["Domain"]["exportVtk"] = True
                #DEM
                data["Domain"]["grain_density"] = obj_domain.grain_density
                data["Domain"]["radius_std"] = obj_domain.radius_std
                data["Domain"]["dem_stiffness"] = obj_domain.dem_stiffness
                data["Domain"]["dem_damping_coefficient"] = obj_domain.dem_damping_coefficient
                data["Domain"]["dem_friction_coefficient"] = obj_domain.dem_friction_coefficient
                data["Domain"]["dem_drag_coefficient"] = obj_domain.dem_drag_coefficient
                data["Domain"]["dem_max_neighbours"] = obj_domain.dem_max_neighbours
                data["Domain"]["bond_modulus"] = obj_domain.bond_modulus
                data["Domain"]["bond_damping_coefficient"] = obj_domain.bond_damping_coefficient
                data["Domain"]["bond_tensile_strength"] = obj_domain.bond_tensile_strength
                data["Domain"]["bond_shear_strength"] = obj_domain.bond_shear_strength
                data["Domain"]["dem_boundary_friction_coefficient"] = obj_domain.dem_boundary_friction_coefficient
                data["Domain"]["dem_boundary_bouncing_coefficient"] = obj_domain.dem_boundary_bouncing_coefficient
                #SOLID
                data["Domain"]["rigid_bouncing_coefficient"] = obj_domain.rigid_bouncing_coefficient
                data["Domain"]["rigid_friction_coefficient"] = obj_domain.rigid_friction_coefficient
                data["Domain"]["rigid_damping_coefficient"] = obj_domain.rigid_damping_coefficient
                
        obj_id = 1
        for obj in bpy.data.objects:
            if obj.doriflow.object_type == 'TYPE_OBSTACLE':
                obj_rigid = obj.doriflow.obstacle
                rigid_body = {
                    "objectId": obj_id, 
                    "blenderName": obj.name,
                    "geometryFile": obj.doriflow.obj_path,
                    "voxelized_points_path": obj.doriflow.voxelized_points_path,
                    "velocity": list(obj_rigid.velocity),
                    "density": obj_rigid.density,
                    "isDynamic": obj_rigid.isFloating,
                    "dimensions": list(obj.dimensions),
                    "rigid_rest_cm": list(obj.location),
                }
                if not obj_rigid.isFloating:
                    min_point = [obj.location[i] - obj.dimensions[i] / 2 for i in range(3)]
                    max_point = [obj.location[i] + obj.dimensions[i] / 2 for i in range(3)]
                    rigid_body["min_point"] = min_point
                    rigid_body["max_point"] = max_point
                if "sphere" in obj.name.lower():
                    rigid_body["radius"] = obj.dimensions[0] / 2  
                elif "cylinder" in obj.name.lower():
                    rigid_body["radius"] = obj.dimensions[0] / 2  
                    rigid_body["height"] = obj.dimensions[2] 
                elif "cube" in obj.name.lower():
                    rigid_body["width"] = obj.dimensions[0] 
                    rigid_body["length"] = obj.dimensions[1] 
                    rigid_body["height"] = obj.dimensions[2]  
                else:
                    rigid_body["dimensions"] = list(obj.dimensions) 
                data["RigidBodies"].append(rigid_body)
                obj_id = obj_id + 1
        for obj in bpy.data.objects:
            if obj.doriflow.object_type == 'TYPE_FLUID':
                obj_fluid = obj.doriflow.fluid
                material_type = obj_fluid.material.lower() 
                if material_type == "liquid":
                    density_value = data["Domain"]["fluid_density"]
                elif material_type == "gas":
                    density_value = data["Domain"]["gas_density"]
                else:
                    density_value = data["Domain"]["fluid_density"]  
                fluid_block = {
                    "objectId": obj_id,  
                    "blenderName": obj.name,
                    "geometryFile": obj.doriflow.obj_path,
                    "voxelized_points_path": obj.doriflow.voxelized_points_path,
                    "velocity": list(obj_fluid.velocity),
                    "density": density_value, 
                    "isDynamic": True,
                    "material": obj_fluid.material,
                }
                data["FluidBlocks"].append(fluid_block)
                obj_id = obj_id + 1

        for obj in bpy.data.objects:
            if obj.doriflow.object_type == 'TYPE_GRAIN':
                obj_grain = obj.doriflow.grain
                grain = {
                    "objectId": obj_id, 
                    "blenderName": obj.name,
                    "geometryFile": obj.doriflow.obj_path,
                    "velocity": list(obj_grain.velocity),
                    # "density": obj_grain.density,
                    "isDynamic": True,
                }
                data["GrainObjects"].append(grain)
                obj_id = obj_id + 1
        data["Domain"]["domainStart"] = format_list(data["Domain"]["domainStart"])
        data["Domain"]["domainEnd"] = format_list(data["Domain"]["domainEnd"])
        data["Domain"]["gravitation"] = format_list(data["Domain"]["gravitation"])
        with open(json_file_path, 'w') as file:
            if len(data["FluidBlocks"]) > 2:
                self.report({'ERROR'}, "Doriflow Demo version only allows up to 2 Fluid objects.\nPlease reduce the number or upgrade to Doriflow Full version.")
                return {'CANCELLED'}

            if len(data["RigidBodies"]) > 2:
                self.report({'ERROR'}, "Doriflow Demo version only allows up to 2 Obstacle objects.\nPlease reduce the number or upgrade to Doriflow Full version.")
                return {'CANCELLED'}

            if len(data["GrainObjects"]) > 2:
                self.report({'ERROR'}, "Doriflow Demo version only allows up to 2 Grain objects.\nPlease reduce the number or upgrade to Doriflow Full version.")
                return {'CANCELLED'}
            json.dump(data, file, cls=CompactJSONEncoder, indent=4)  # Assuming 'data' is your JSON-serializable Python object
        self.report({'INFO'}, f"Doriflow Configuration exported to {json_file_path}")
        return {'FINISHED'}

def register():
    bpy.utils.register_class(DORIFLOW_OT_ExportData)

def unregister():
    bpy.utils.unregister_class(DORIFLOW_OT_ExportData)

if __name__ == "__main__":
    register()