# Doriflow Engine - Fluid Simulation for Blender 3D
# Copyright (C) 2024 Doriflow Team
# This software is licensed under the Creative Commons Attribution-NonCommercial 4.0 International License (CC BY-NC 4.0).

# You are free to:
# -Share: Copy and redistribute the material in any medium or format.
# -Adapt: Remix, transform, and build upon the material.

# UNDER THE FOLOWING TERMS:
# -Attribution: You must give appropriate credit, provide a link to the license, and indicate if changes were made.
# -Appropriate credit should include the following:
#   -The original author's name: Doriflow Team
#   -A link to the original source (if applicable).
#   -A link to the full license: https://creativecommons.org/licenses/by-nc/4.0/.
#   -A clear indication of any changes made, such as: "This material has been modified."
# NonCommercial: You may not use the material for commercial purposes.
# Disclaimer:
# -This simulation engine is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. In no event shall the authors be liable for any claim, damages, or other liability arising from the use of this software.

# For more details, refer to the full license text at:
# https://creativecommons.org/licenses/by-nc/4.0/.

#-----------------------------------------------------------------------------------------------------------------------#
import bpy
from bpy.types import Operator
from bpy.types import PropertyGroup
from bpy.props import StringProperty
import os
import mathutils
from ..utils import calculate_bounding_box

class DORIFLOW_OT_AssignObstacle(Operator):
    bl_idname = "doriflow.assign_obstacle"
    bl_label = "Assign Obstacle"
    bl_options = {'REGISTER'}
    bl_description = "Assign selected objects as Obstacles"
    def execute(self, context):
        for obj in context.selected_objects:
            obstacle_objects = [obj for obj in bpy.data.objects if obj.doriflow.object_type == 'TYPE_OBSTACLE']
            if len(obstacle_objects) >= 2:
                self.report({'WARNING'}, "Doriflow Demo version only allows max 2 Obstacle objects.\nPurchase Doriflow Full version for unlimited assignments.")
                return {'CANCELLED'}
            if obj.type == 'MESH':  
                obj.doriflow.object_type = 'TYPE_OBSTACLE'
                bpy.context.view_layer.objects.active = obj
                bpy.context.object.display_type = 'SOLID'
        self.report({'INFO'}, f"Assigned {len(context.selected_objects)} objects as Obstacles")
        return {'FINISHED'}
class DORIFLOW_OT_AssignKeyframedObstacle(Operator):
    bl_idname = "doriflow.assign_keyframed_obstacle"
    bl_label = "Assign Keyframed Obstacle"
    bl_options = {'REGISTER'}
    bl_description = "Assign selected objects as Keyframed Obstacles"
class DORIFLOW_OT_AssignFluid(Operator):
    bl_idname = "doriflow.assign_fluid"
    bl_label = "Assign Fluid"
    bl_options = {'REGISTER'}
    bl_description = "Assign selected objects as Fluid"
    def execute(self, context):
        for obj in context.selected_objects:
            fluid_objects = [obj for obj in bpy.data.objects if obj.doriflow.object_type == 'TYPE_FLUID']
            if len(fluid_objects) >= 2:
                self.report({'WARNING'}, "Doriflow Demo version only allows max 2 Fluid objects.\nPurchase Doriflow Full version for unlimited assignments.")
                return {'CANCELLED'}
            if obj.type == 'MESH':
                obj.doriflow.object_type = 'TYPE_FLUID'
                bpy.context.view_layer.objects.active = obj
                bpy.context.object.display_type = 'WIRE'
        self.report({'INFO'}, f"Assigned {len(context.selected_objects)} objects as Fluid")
        return {'FINISHED'}
import bpy

class DORIFLOW_OT_AssignInlet(Operator):
    bl_idname = "doriflow.assign_inlet"
    bl_label = "Assign Inlet"
    bl_options = {'REGISTER'}
    bl_description = "Assign selected objects as Inlet"

class DORIFLOW_OT_AssignOutlet(Operator):
    bl_idname = "doriflow.assign_outlet"
    bl_label = "Assign Outlet"
    bl_options = {'REGISTER'}
    bl_description = "Assign selected objects as Outlet"
class DORIFLOW_OT_AssignDomain(Operator):
    bl_idname = "doriflow.assign_domain"
    bl_label = "Assign Domain"
    bl_options = {'REGISTER'}
    bl_description = "Assign selected object as Domain"
    def execute(self, context):
        for obj in context.selected_objects:
            if obj.type == 'MESH':
                obj.doriflow.object_type = 'TYPE_DOMAIN'
                bpy.context.view_layer.objects.active = obj
                bpy.context.object.display_type = 'WIRE'
        self.report({'INFO'}, "Assigned selected object as Domain")
        return {'FINISHED'}
    
class DORIFLOW_OT_AssignGrain(Operator):
    bl_idname = "doriflow.assign_grain"
    bl_label = "Assign Grain"
    bl_options = {'REGISTER'}
    bl_description = "Assign selected objects as Grain"
    def execute(self, context):
        for obj in context.selected_objects:
            grain_objects = [obj for obj in bpy.data.objects if obj.doriflow.object_type == 'TYPE_GRAIN']
            if len(grain_objects) >= 2:
                self.report({'WARNING'}, "Doriflow Demo version only allows max 2 Grain objects.\nPurchase Doriflow Full version for unlimited assignments.")
                return {'CANCELLED'}
            if obj.type == 'MESH':
                obj.doriflow.object_type = 'TYPE_GRAIN'
                bpy.context.view_layer.objects.active = obj
                bpy.context.object.display_type = 'WIRE'
        self.report({'INFO'}, f"Assigned {len(context.selected_objects)} objects as Grain")
        return {'FINISHED'}
class DORIFLOW_OT_AssignNone(Operator):
    bl_idname = "doriflow.assign_none"
    bl_label = "Assign None"
    bl_options = {'REGISTER'}
    bl_description = "Assign selected objects as None"
    def execute(self, context):
        for obj in context.selected_objects:
            obj.doriflow.object_type = 'TYPE_NONE'
        self.report({'INFO'}, f"Assigned {len(context.selected_objects)} objects as None")
        return {'FINISHED'}
class DORIFLOW_OT_ExportDomain(Operator):
    bl_idname = "doriflow.export_domain"
    bl_label = "Export Domain As Obj File"
    bl_options = {'REGISTER'}
    bl_description = "Export Domain object as obj file"
    def execute(self, context):
        domain_objects = [obj for obj in bpy.data.objects if obj.doriflow.object_type == 'TYPE_DOMAIN']
        if len(domain_objects) > 1:
            self.report({'ERROR'}, "More than one domain object found. Only one is allowed.")
            return {'CANCELLED'}
        elif len(domain_objects) == 0:
            self.report({'ERROR'}, "No domain object found.")
            return {'CANCELLED'}
        else:
            domain_object = domain_objects[0]
            bpy.context.view_layer.objects.active = domain_object
            domain_object_copy = domain_object.copy()
            domain_object_copy.data = domain_object.data.copy()
            bpy.context.collection.objects.link(domain_object_copy)
            blender_file_path = bpy.data.filepath
            blender_file_dir = os.path.dirname(blender_file_path)
            blender_file_name = os.path.splitext(os.path.basename(blender_file_path))[0]
            cache_folder_name = f"{blender_file_name}_cache"
            cache_folder_path = os.path.join(blender_file_dir, cache_folder_name)
            domain_file_path = os.path.join(cache_folder_path, "domain.obj")
            bpy.ops.object.select_all(action='DESELECT')
            domain_object_copy.select_set(True)
            bpy.ops.wm.obj_export(filepath=domain_file_path, export_selected_objects=True, export_materials=False, forward_axis="Y", up_axis="Z")
            context.object.doriflow.obj_path = domain_file_path
            self.report({'INFO'}, f"Domain object exported to {domain_file_path}")
            bpy.ops.object.select_all(action='DESELECT')
            domain_object_copy.select_set(True)
            bpy.ops.object.delete()
            bpy.context.view_layer.objects.active = domain_object
            domain_object.select_set(True)
            return {'FINISHED'}
class DORIFLOW_OT_ExportFluid(bpy.types.Operator):
    bl_idname = "doriflow.export_fluid"
    bl_label = "Export Fluids As Obj Files"
    bl_options = {'REGISTER'}
    bl_description = "Export Fluid objects as obj files"
    def execute(self, context):
        fluids = [obj for obj in bpy.data.objects if obj.doriflow.object_type == 'TYPE_FLUID']
        if len(fluids) > 2:
            self.report({'ERROR'}, "Doriflow Demo version only allows up to 2 Fluid objects.\nPurchase Doriflow Full version for unlimited objects.")
            return {'CANCELLED'}

        export_count = 0
        base_filename = "fluid_{}.obj"
        blender_file_path = bpy.data.filepath
        blender_file_dir = os.path.dirname(blender_file_path)
        cache_folder_name = f"{os.path.splitext(os.path.basename(blender_file_path))[0]}_cache"
        cache_folder_path = os.path.join(blender_file_dir, cache_folder_name)
        obstacles = [obj for obj in bpy.data.objects if obj.doriflow.object_type == 'TYPE_OBSTACLE']
        fluids = [obj for obj in bpy.data.objects if obj.doriflow.object_type == 'TYPE_FLUID']
        original_scales = {obstacle: obstacle.scale.copy() for obstacle in obstacles}
        for fluid in fluids:
            export_count += 1
            filename = base_filename.format(export_count)
            filepath = os.path.join(cache_folder_path, filename)
            bpy.ops.object.select_all(action='DESELECT')
            fluid.select_set(True)
            bpy.context.view_layer.objects.active = fluid
            bpy.ops.wm.obj_export(filepath=filepath, export_selected_objects=True, export_materials=False, forward_axis="Y", up_axis="Z")
            fluid.doriflow.obj_path = filepath
            voxelized_points_filename = f"{fluid.name}_voxelized_points.txt"
            voxelized_points_path = os.path.join(cache_folder_path, voxelized_points_filename)
            fluid.doriflow.voxelized_points_path = voxelized_points_path
            self.report({'INFO'}, f"Exported Modified Fluid to {filepath}")
        if export_count == 0:
            self.report({'WARNING'}, "No fluid objects found for export.")
            return {'CANCELLED'}
        for obj in bpy.data.objects:
            if obj.doriflow.object_type == 'TYPE_DOMAIN':
                domain_object = obj
                bpy.context.view_layer.objects.active = domain_object
                domain_object.select_set(True)
        return {'FINISHED'}
class DORIFLOW_OT_ExportGrain(Operator):
    bl_idname = "doriflow.export_grain"
    bl_label = "Export Grains As Obj Files"
    bl_options = {'REGISTER'}
    bl_description = "Export Grain objects as obj files"
    def execute(self, context):
        grains = [obj for obj in bpy.data.objects if obj.doriflow.object_type == 'TYPE_GRAIN']
        if len(grains) > 2:
            self.report({'ERROR'}, "Doriflow Demo version only allows up to 2 Grain objects.\nPurchase Doriflow Full version for unlimited objects.")
            return {'CANCELLED'}

        export_count = 0
        base_filename = "grain_{}.obj"
        blender_file_path = bpy.data.filepath
        blender_file_dir = os.path.dirname(blender_file_path)
        cache_folder_name = f"{os.path.splitext(os.path.basename(blender_file_path))[0]}_cache"
        cache_folder_path = os.path.join(blender_file_dir, cache_folder_name)
        grains = [obj for obj in bpy.data.objects if obj.doriflow.object_type == 'TYPE_GRAIN']
        for grain in grains:
            export_count += 1
            filename = base_filename.format(export_count)
            filepath = os.path.join(cache_folder_path, filename)
            bpy.ops.object.select_all(action='DESELECT')
            grain.select_set(True)
            bpy.context.view_layer.objects.active = grain
            bpy.ops.wm.obj_export(filepath=filepath, export_selected_objects=True, export_materials=False, forward_axis="Y", up_axis="Z")
            grain.doriflow.obj_path = filepath
            self.report({'INFO'}, f"Exported Grain to {filepath}")
        if export_count == 0:
            self.report({'WARNING'}, "No grain objects found for export.")
            return {'CANCELLED'}
        for obj in bpy.data.objects:
            if obj.doriflow.object_type == 'TYPE_DOMAIN':
                domain_object = obj
                bpy.context.view_layer.objects.active = domain_object
                domain_object.select_set(True)
        return {'FINISHED'}
class DORIFLOW_OT_ExportInlet(bpy.types.Operator):
    bl_idname = "doriflow.export_inlet"
    bl_label = "Export Inlets As Obj Files"
    bl_options = {'REGISTER'}
    bl_description = "Export Inlet objects as obj files"
class DORIFLOW_OT_ExportOutlet(bpy.types.Operator):
    bl_idname = "doriflow.export_outlet"
    bl_label = "Export Outlets As Obj Files"
    bl_options = {'REGISTER'}
    bl_description = "Export Outlet objects as obj files"
class DORIFLOW_OT_ExportObstacle(Operator):
    bl_idname = "doriflow.export_obstacle"
    bl_label = "Export Objects As Obj File"
    bl_options = {'REGISTER'}
    bl_description = "Export Obstacle objects as obj files"
    def execute(self, context):
        obstacles = [obj for obj in bpy.data.objects if obj.doriflow.object_type == 'TYPE_OBSTACLE']
        if len(obstacles) > 2:
            self.report({'ERROR'}, "Doriflow Demo version only allows up to 2 Obstacle objects.\nPurchase Doriflow Full version for unlimited objects.")
            return {'CANCELLED'}
        export_count = 0
        base_filename = "obstacle_{}.obj" 
        blender_file_path = bpy.data.filepath
        blender_file_dir = os.path.dirname(blender_file_path)
        blender_file_name = os.path.splitext(os.path.basename(blender_file_path))[0]
        cache_folder_name = f"{blender_file_name}_cache"
        cache_folder_path = os.path.join(blender_file_dir, cache_folder_name)
        keywords = ['Cube', 'Sphere', 'Cylinder','cube', 'sphere', 'cylinder']
        for obj in context.scene.objects:
            if obj.doriflow.object_type == 'TYPE_OBSTACLE':
                export_count += 1
                keyword = next((k for k in keywords if k in obj.name), '')
                formatted_obj_name = keyword + '_' + str(export_count)
                filename = base_filename.format(formatted_obj_name)
                filepath = os.path.join(cache_folder_path, filename)
                bpy.ops.object.select_all(action='DESELECT')
                obj.select_set(True)
                bpy.ops.wm.obj_export(filepath=filepath, export_selected_objects=True, export_materials=False, forward_axis="Y", up_axis="Z")
                obj.doriflow.obj_path = filepath
                voxelize_points_filename = f"{obj.name}_voxelized_points.txt"
                voxelized_points_path = os.path.join(cache_folder_path, voxelize_points_filename)
                obj.doriflow.voxelized_points_path = voxelized_points_path
                self.report({'INFO'}, f"Exported Obstacle to {filepath}")
        if export_count == 0:
            self.report({'WARNING'}, "No obstacle objects found for export.")
            return {'CANCELLED'}
        for obj in bpy.data.objects:
            if obj.doriflow.object_type == 'TYPE_DOMAIN':
                domain_object = obj
                bpy.context.view_layer.objects.active = domain_object
                domain_object.select_set(True)
        return {'FINISHED'}
class DORIFLOW_OT_ExportKeyframedObstacle(Operator):
    bl_idname = "doriflow.export_keyframed_obstacle"
    bl_label = "Export Keyframed Objects As Obj File"
    bl_options = {'REGISTER'}
    bl_description = "Export Keyframed Obstacle objects as obj files"
    
class DORIFLOW_OT_Add(bpy.types.Operator):
    bl_idname = "doriflow.doriflow_add"
    bl_label = "Add Doriflow object"
    bl_description = "Add active object"
    bl_options = {'REGISTER'}
    @classmethod
    def poll(csl, context):
        return True
    def execute(self, context):
        obj = bpy.context.active_object if context is None else context.active_object
        for obj in bpy.context.selected_objects:
            obj.doriflow.is_active = True
        return {'FINISHED'}
class DORIFLOW_OT_Remove(bpy.types.Operator):
    bl_idname = "doriflow.doriflow_remove"
    bl_label = "Remove Doriflow object"
    bl_description = "Remove settings from Object"
    bl_options = {'REGISTER'}
    @classmethod
    def poll(csl, context):
        return True
    def execute(self, context):
        obj = bpy.context.active_object if context is None else context.active_object
        obj.doriflow.object_type = 'TYPE_NONE'
        obj.doriflow.is_active = False
        return {'FINISHED'}
def register():
    bpy.utils.register_class(DORIFLOW_OT_Add)
    bpy.utils.register_class(DORIFLOW_OT_Remove)
    bpy.utils.register_class(DORIFLOW_OT_AssignDomain)
    bpy.utils.register_class(DORIFLOW_OT_AssignFluid)
    bpy.utils.register_class(DORIFLOW_OT_AssignObstacle)
    bpy.utils.register_class(DORIFLOW_OT_AssignKeyframedObstacle)
    bpy.utils.register_class(DORIFLOW_OT_AssignInlet)
    bpy.utils.register_class(DORIFLOW_OT_AssignOutlet)
    bpy.utils.register_class(DORIFLOW_OT_AssignGrain)
    bpy.utils.register_class(DORIFLOW_OT_AssignNone)
    bpy.utils.register_class(DORIFLOW_OT_ExportDomain)
    bpy.utils.register_class(DORIFLOW_OT_ExportFluid)
    bpy.utils.register_class(DORIFLOW_OT_ExportObstacle)
    bpy.utils.register_class(DORIFLOW_OT_ExportKeyframedObstacle)
    bpy.utils.register_class(DORIFLOW_OT_ExportInlet)
    bpy.utils.register_class(DORIFLOW_OT_ExportOutlet)
    bpy.utils.register_class(DORIFLOW_OT_ExportGrain)



def unregister():
    bpy.utils.unregister_class(DORIFLOW_OT_AssignDomain)
    bpy.utils.unregister_class(DORIFLOW_OT_AssignFluid)
    bpy.utils.unregister_class(DORIFLOW_OT_AssignObstacle)
    bpy.utils.unregister_class(DORIFLOW_OT_AssignKeyframedObstacle)
    bpy.utils.unregister_class(DORIFLOW_OT_AssignInlet)
    bpy.utils.unregister_class(DORIFLOW_OT_AssignOutlet)
    bpy.utils.unregister_class(DORIFLOW_OT_AssignGrain)
    bpy.utils.unregister_class(DORIFLOW_OT_AssignNone)
    bpy.utils.unregister_class(DORIFLOW_OT_ExportDomain)
    bpy.utils.unregister_class(DORIFLOW_OT_ExportFluid)
    bpy.utils.unregister_class(DORIFLOW_OT_ExportObstacle)
    bpy.utils.unregister_class(DORIFLOW_OT_ExportKeyframedObstacle)
    bpy.utils.unregister_class(DORIFLOW_OT_ExportInlet)
    bpy.utils.unregister_class(DORIFLOW_OT_ExportOutlet)
    bpy.utils.unregister_class(DORIFLOW_OT_ExportGrain)
    bpy.utils.unregister_class(DORIFLOW_OT_Add)
    bpy.utils.unregister_class(DORIFLOW_OT_Remove)

if __name__ == "__main__":
    register()
