# Doriflow Engine - Fluid Simulation for Blender 3D
# Copyright (C) 2024 Doriflow Team
# This software is licensed under the Creative Commons Attribution-NonCommercial 4.0 International License (CC BY-NC 4.0).

# You are free to:
# -Share: Copy and redistribute the material in any medium or format.
# -Adapt: Remix, transform, and build upon the material.

# UNDER THE FOLOWING TERMS:
# -Attribution: You must give appropriate credit, provide a link to the license, and indicate if changes were made.
# -Appropriate credit should include the following:
#   -The original author's name: Doriflow Team
#   -A link to the original source (if applicable).
#   -A link to the full license: https://creativecommons.org/licenses/by-nc/4.0/.
#   -A clear indication of any changes made, such as: "This material has been modified."
# NonCommercial: You may not use the material for commercial purposes.
# Disclaimer:
# -This simulation engine is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. In no event shall the authors be liable for any claim, damages, or other liability arising from the use of this software.

# For more details, refer to the full license text at:
# https://creativecommons.org/licenses/by-nc/4.0/.

#-----------------------------------------------------------------------------------------------------------------------#
import bpy
import os
import json
from pathlib import Path
from bpy.props import EnumProperty, PointerProperty
from bpy.types import PropertyGroup, Operator
from mathutils import Matrix

def apply_preset(preset_path, context):
    with open(preset_path, 'r') as file:
        preset_data = json.load(file)
    
    # Domain Settings
    for obj in bpy.data.objects:
        if obj.doriflow.object_type == 'TYPE_DOMAIN':
            obj_domain = obj.doriflow.domain
            domain_data = preset_data.get("Domain", {})
            obj_domain.timestep_per_frame = domain_data.get("timestep_per_frame", obj_domain.timestep_per_frame)
            obj_domain.resolution = domain_data.get("resolution", obj_domain.resolution)
            obj_domain.gravitation = domain_data.get("gravitation", obj_domain.gravitation)
            obj_domain.time_step_size = domain_data.get("timeStepSize", obj_domain.time_step_size)*1000
            obj_domain.stiffness = domain_data.get("stiffness", obj_domain.stiffness)
            obj_domain.exponent = domain_data.get("exponent", obj_domain.exponent)
            obj_domain.BCs = domain_data.get("BCs", obj_domain.BCs)
            obj_domain.viscosity = domain_data.get("viscosity", obj_domain.viscosity)
            obj_domain.surface_tension = domain_data.get("surface_tension", obj_domain.surface_tension)
            obj_domain.boundary_collision_factor = domain_data.get("boundary_collision_factor", obj_domain.boundary_collision_factor)
            obj_domain.rotation_factor = domain_data.get("rotation_factor", obj_domain.rotation_factor)
            obj_domain.export_fluid_velocity = domain_data.get("export_fluid_velocity", obj_domain.export_fluid_velocity)
            obj_domain.export_rigid_pressure_force = domain_data.get("export_rigid_pressure_force", obj_domain.export_rigid_pressure_force)
            obj_domain.export_rigid_viscous_force = domain_data.get("export_rigid_viscous_force", obj_domain.export_rigid_viscous_force)

    # Fluid Blocks
    for obj in bpy.data.objects:
        if obj.doriflow.object_type == 'TYPE_FLUID':
            obj_fluid = obj.doriflow.fluid
            fluid_blocks = preset_data.get("FluidBlocks", [])

            for fluid_data in fluid_blocks:
                if obj.name == fluid_data.get("blenderName"):
                    obj_fluid.density = fluid_data.get("density", obj_fluid.density)
                    obj_fluid.velocity = fluid_data.get("velocity", obj_fluid.velocity)

    # Rigid Bodies
    for obj in bpy.data.objects:
        if obj.doriflow.object_type == 'TYPE_OBSTACLE':
            obj_obstacle = obj.doriflow.obstacle
            rigid_bodies = preset_data.get("RigidBodies", [])

            for obstacle_data in rigid_bodies:
                if obj.name == obstacle_data.get("blenderName"):
                    obj_obstacle.density = obstacle_data.get("density", obj_obstacle.density)
                    obj_obstacle.velocity = obstacle_data.get("velocity", obj_obstacle.velocity)
                    obj_obstacle.isFloating = obstacle_data.get("isDynamic", obj_obstacle.isFloating)
                    bpy.ops.object.origin_set(type='ORIGIN_CENTER_OF_VOLUME', center='MEDIAN')
    
    # Grains
    for obj in bpy.data.objects:
        if obj.doriflow.object_type == 'TYPE_GRAIN':
            obj_grain = obj.doriflow.grain
            grains = preset_data.get("Grains", [])

            for grain_data in grains:
                if obj.name == grain_data.get("blenderName"):
                    obj_grain.velocity = grain_data.get("velocity", obj_grain.velocity)
                    bpy.ops.object.origin_set(type='ORIGIN_CENTER_OF_VOLUME', center='MEDIAN')

class ImportPresetObjects(Operator):
    bl_idname = "doriflow.import_preset_objects"
    bl_label = "Import Preset Objects"
    bl_description = "Click to apply the selected preset."

    def execute(self, context):
        current_file_path = os.path.realpath(__file__)
        current_directory = Path(current_file_path).parent
        default_presets_folder_path = current_directory.parent / "presets"
        user_presets_folder_path = bpy.path.abspath(context.scene.preset_properties.presets_folder_path)
        preset_name = context.scene.preset_properties.presets_scenes
        preset_path = None
        if (default_presets_folder_path / preset_name).exists():
            preset_path = default_presets_folder_path / preset_name
        elif os.path.exists(user_presets_folder_path) and os.path.isdir(user_presets_folder_path):
            if (Path(user_presets_folder_path) / preset_name).exists():
                preset_path = Path(user_presets_folder_path) / preset_name
        if preset_path is None:
            self.report({'ERROR'}, f"Preset '{preset_name}' not found in either default or user-defined folders.")
            return {'CANCELLED'}
        
        fluid_count = 1
        solid_count = 1
        inlet_count = 1
        keyframed_count = 1
        grain_count = 1
        
        for obj_file in os.listdir(preset_path):
            if obj_file.endswith(".obj"):
                bpy.ops.wm.obj_import(filepath=os.path.join(preset_path, obj_file), forward_axis='Y', up_axis='Z')
                imported_obj = bpy.context.selected_objects[0]
                keywords = ['Cube', 'Sphere', 'Cylinder']
                if obj_file.startswith('domain'):
                    bpy.ops.object.select_all(action='DESELECT')
                    imported_obj.select_set(True)
                    bpy.ops.doriflow.assign_domain()
                    bpy.ops.object.select_all(action='DESELECT')
                    imported_obj.select_set(True)
                    bpy.context.view_layer.objects.active = imported_obj
                    imported_obj.display_type = 'WIRE'  
                    imported_obj.name = "DF.Domain"
                elif obj_file.startswith('fluid'):
                    bpy.ops.object.select_all(action='DESELECT')
                    imported_obj.select_set(True)
                    bpy.ops.doriflow.assign_fluid()
                    bpy.ops.object.select_all(action='DESELECT')
                    imported_obj.select_set(True)
                    bpy.context.view_layer.objects.active = imported_obj
                    imported_obj.display_type = 'WIRE' 
                    imported_obj.name = f"DF.Fluid.{fluid_count:02}"
                    fluid_count += 1
                elif obj_file.startswith('obstacle'):
                    bpy.ops.object.select_all(action='DESELECT')
                    imported_obj.select_set(True)
                    bpy.ops.doriflow.assign_obstacle()
                    keyword = next((k for k in keywords if k in imported_obj.name), '')
                    if keyword:
                        imported_obj.name = f"DF.Solid.{keyword}.{solid_count:02}"
                    else:
                        imported_obj.name = f"DF.Solid.{solid_count:02}"
                    solid_count += 1
                elif obj_file.startswith('grain'):
                    bpy.ops.object.select_all(action='DESELECT')
                    imported_obj.select_set(True)
                    bpy.ops.doriflow.assign_grain()
                    bpy.ops.object.select_all(action='DESELECT')
                    imported_obj.select_set(True)
                    bpy.context.view_layer.objects.active = imported_obj
                    imported_obj.display_type = 'WIRE' 
                    imported_obj.name = f"DF.Grain.{grain_count:02}"
                    grain_count += 1
                if not imported_obj.doriflow.is_active:
                    bpy.ops.doriflow.doriflow_add()

        bpy.ops.object.select_all(action='SELECT')
        bpy.ops.object.origin_set(type='ORIGIN_CENTER_OF_VOLUME', center='MEDIAN')
        for obj in bpy.data.objects:
            if obj.doriflow.object_type == 'TYPE_DOMAIN':
                domain_object = obj
                bpy.context.view_layer.objects.active = domain_object
                domain_object.select_set(True)  
        preset_json_path = os.path.join(preset_path, f"{preset_name}_initial_condition.json")
        if os.path.exists(preset_json_path):
            apply_preset(preset_json_path, context)
            self.report({'INFO'}, f"Preset '{preset_name}' applied successfully.")
        else:
            self.report({'WARNING'}, f"Preset JSON file '{preset_json_path}' not found.")
        
        return {'FINISHED'}

def register():
    bpy.utils.register_class(ImportPresetObjects)   

def unregister():    
    bpy.utils.unregister_class(ImportPresetObjects)

