# Doriflow Engine - Fluid Simulation for Blender 3D
# Copyright (C) 2024 Doriflow Team
# This software is licensed under the Creative Commons Attribution-NonCommercial 4.0 International License (CC BY-NC 4.0).

# You are free to:
# -Share: Copy and redistribute the material in any medium or format.
# -Adapt: Remix, transform, and build upon the material.

# UNDER THE FOLOWING TERMS:
# -Attribution: You must give appropriate credit, provide a link to the license, and indicate if changes were made.
# -Appropriate credit should include the following:
#   -The original author's name: Doriflow Team
#   -A link to the original source (if applicable).
#   -A link to the full license: https://creativecommons.org/licenses/by-nc/4.0/.
#   -A clear indication of any changes made, such as: "This material has been modified."
# NonCommercial: You may not use the material for commercial purposes.
# Disclaimer:
# -This simulation engine is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. In no event shall the authors be liable for any claim, damages, or other liability arising from the use of this software.

# For more details, refer to the full license text at:
# https://creativecommons.org/licenses/by-nc/4.0/.

#-----------------------------------------------------------------------------------------------------------------------#
import bpy
import os
import numpy as np
from ..utils import calculate_bounding_box
import math

def calculate_grid_properties(largest_domain_size, resolution):
        grid_size = largest_domain_size / resolution
        particle_radius = round(grid_size / 4,4)
        particle_diameter = 2 * particle_radius
        grid_num=1
        
        return grid_size, particle_diameter, grid_num, particle_radius

class DORIFLOW_OT_VoxelizationGeometryNodes(bpy.types.Operator):
    bl_idname = "doriflow.voxelization_geometry_nodes"
    bl_label = "Voxelization Geometry Nodes"
    bl_description = (
                    "Initialize the simulation with particles for the selected resolution. "
                    "Be cautious with extremely high resolutions as they may freeze the system. "
                    "Increase the resolution if there are not enough particles or if fluid leaks through solid walls. "
                    "Decrease the resolution if the system runs out of memory or becomes LAGGING.")
    
    def write_vtk_points_polydata_legacy_binary(self,
        file_path,
        points,
        vectors_dict=None,
        scalars_dict=None
    ):
        if vectors_dict is None:
            vectors_dict = {}
        if scalars_dict is None:
            scalars_dict = {}
        num_points = len(points)
        points_f32 = points.astype(np.float32)
        points_be = points_f32.byteswap().tobytes()
        connectivity = np.zeros((num_points, 2), dtype=np.int32)
        for i in range(num_points):
            connectivity[i, 0] = 1  
            connectivity[i, 1] = i  
        connectivity_be = connectivity.byteswap().tobytes()
        with open(file_path, "wb") as f:
            header = (
                "# vtk DataFile Version 3.0\n"
                "Binary point cloud data\n"
                "BINARY\n"
                "DATASET POLYDATA\n"
            )
            f.write(header.encode("utf-8"))
            f.write(f"POINTS {num_points} float\n".encode("utf-8"))
            f.write(points_be)
            f.write(f"VERTICES {num_points} {2 * num_points}\n".encode("utf-8"))
            f.write(connectivity_be)
            f.write(f"POINT_DATA {num_points}\n".encode("utf-8"))
            for name, data in scalars_dict.items():
                if len(data) != num_points:
                    raise ValueError(f"Scalar array length mismatch for '{name}'")
                f.write(f"SCALARS {name} float 1\nLOOKUP_TABLE default\n".encode("utf-8"))
                data_f32 = data.astype(np.float32)
                data_be = data_f32.byteswap().tobytes()
                f.write(data_be)
            # --- Write Vectors ---
            for name, data in vectors_dict.items():
                if len(data) != num_points:
                    raise ValueError(f"Vector array length mismatch for '{name}'")
                f.write(f"VECTORS {name} float\n".encode("utf-8"))
                data_f32 = data.astype(np.float32)
                data_be = data_f32.byteswap().tobytes()
                f.write(data_be)

    def delete_existing_geometry_node_group(self, obj):
        node_group_names = [
            f"DF.GeometryNodes_{obj.name}",
            f"DF.GeometryNodes_Fluid_{obj.name}"]
        for node_group_name in node_group_names:
            if node_group_name in bpy.data.node_groups:
                bpy.data.node_groups.remove(bpy.data.node_groups[node_group_name], do_unlink=True)
    def create_geometry_node_system_for_fluid(self, obj, particle_diameter,fluid_obj, domain_obj,obstacles=None, keyframed_obstacles=None):
        self.delete_existing_geometry_node_group(obj)
        node_group_name = f"DF.GeometryNodes_Fluid_{obj.name}"
        if node_group_name in bpy.data.node_groups:
            node_group = bpy.data.node_groups[node_group_name]
        else:
            node_group = bpy.data.node_groups.new(name=node_group_name, type='GeometryNodeTree')
            
            group_input = node_group.nodes.new('NodeGroupInput')
            group_output = node_group.nodes.new('NodeGroupOutput')
            object_info_domain = node_group.nodes.new('GeometryNodeObjectInfo')
            transform_scale = node_group.nodes.new('GeometryNodeTransform')
            mesh_boolean_intersect = node_group.nodes.new('GeometryNodeMeshBoolean')
            mesh_boolean_difference = node_group.nodes.new('GeometryNodeMeshBoolean')
            join_geometry_obstacles = node_group.nodes.new('GeometryNodeJoinGeometry')
            transform_geometry = node_group.nodes.new('GeometryNodeTransform')
            mesh_to_volume = node_group.nodes.new('GeometryNodeMeshToVolume')
            distribute_points = node_group.nodes.new('GeometryNodeDistributePointsInVolume')
            points_to_vertices = node_group.nodes.new('GeometryNodePointsToVertices')
            value_node = node_group.nodes.new('ShaderNodeValue')
            
            object_info_obstacles = []
            if obstacles:
                for i, obstacle in enumerate(obstacles):
                    obj_info = node_group.nodes.new('GeometryNodeObjectInfo')
                    obj_info.inputs['Object'].default_value = obstacle
                    obj_info.transform_space = 'RELATIVE'
                    object_info_obstacles.append(obj_info)
            
            object_info_keyframed = []
            if keyframed_obstacles:
                for i, keyframed_obstacle in enumerate(keyframed_obstacles):
                    obj_info = node_group.nodes.new('GeometryNodeObjectInfo')
                    obj_info.inputs['Object'].default_value = keyframed_obstacle
                    obj_info.transform_space = 'RELATIVE'
                    object_info_keyframed.append(obj_info)
            
            group_input.location = (-1600, 0)
            object_info_domain.location = (-1400, 300)
            mesh_boolean_intersect.location = (-1000, 0)
            join_geometry_obstacles.location = (-1000, -300)
            mesh_boolean_difference.location = (-600, 0)
            transform_geometry.location = (-400, 0)
            transform_scale.location = (-800, 300)
            mesh_to_volume.location = (-200, 0)
            distribute_points.location = (0, 0)
            points_to_vertices.location = (200, 0)
            group_output.location = (400, 0)
            value_node.location = (-400, -200)
            
            grid_size = float(particle_diameter*2)
            object_info_domain.inputs['Object'].default_value = domain_obj
            object_info_domain.transform_space = 'RELATIVE'
            domain_obj_copy = domain_obj.copy()
            domain_obj_copy.data = domain_obj.data.copy()
            bpy.context.collection.objects.link(domain_obj_copy)
            bpy.ops.object.select_all(action='DESELECT')
            domain_obj_copy.select_set(True)
            bpy.ops.object.transform_apply(location=True, rotation=True, scale=True)
            bpy.ops.object.origin_set(type='ORIGIN_GEOMETRY', center='MEDIAN')
            dimensions = domain_obj_copy.dimensions
            shrink_factor_x = 1.0 - (grid_size / dimensions.x)
            shrink_factor_y = 1.0 - (grid_size / dimensions.y)
            shrink_factor_z = 1.0 - (grid_size / dimensions.z)
            
            mesh_boolean_intersect.operation = 'INTERSECT'
            mesh_boolean_difference.operation = 'DIFFERENCE'
            mesh_to_volume.resolution_mode = "VOXEL_SIZE"
            mesh_to_volume.inputs['Voxel Size'].default_value = 0.1
            distribute_points.mode = 'DENSITY_GRID'
            distribute_points.inputs['Spacing'].default_value[0] = particle_diameter/obj.scale.x
            distribute_points.inputs['Spacing'].default_value[1] = particle_diameter/obj.scale.y
            distribute_points.inputs['Spacing'].default_value[2] = particle_diameter/obj.scale.z
            distribute_points.inputs['Threshold'].default_value = 0.1
            value_node.outputs[0].default_value = 0.1
            
            transform_scale.inputs['Scale'].default_value[0] = shrink_factor_x
            transform_scale.inputs['Scale'].default_value[1] = shrink_factor_y
            transform_scale.inputs['Scale'].default_value[2] = shrink_factor_z
            
            transform_geometry.inputs['Translation'].default_value = obj.location
            transform_geometry.inputs['Scale'].default_value = obj.scale
            transform_geometry.inputs['Rotation'].default_value = obj.rotation_euler
            
            node_group.interface.new_socket(name="Geometry", description="Geometry input", in_out='INPUT', socket_type='NodeSocketGeometry')
            node_group.interface.new_socket(name="Geometry", description="Geometry output", in_out='OUTPUT', socket_type='NodeSocketGeometry')
            
            node_group.links.new(group_input.outputs['Geometry'], mesh_boolean_intersect.inputs[1])
            node_group.links.new(object_info_domain.outputs['Geometry'],transform_scale.inputs['Geometry'])
            node_group.links.new(transform_scale.outputs['Geometry'], mesh_boolean_intersect.inputs[1])
            
            for obj_info in object_info_obstacles:
                node_group.links.new(obj_info.outputs['Geometry'], join_geometry_obstacles.inputs['Geometry'])
            for obj_info in object_info_keyframed:
                node_group.links.new(obj_info.outputs['Geometry'], join_geometry_obstacles.inputs['Geometry'])
            
            node_group.links.new(mesh_boolean_intersect.outputs['Mesh'], mesh_boolean_difference.inputs[0])
            node_group.links.new(join_geometry_obstacles.outputs['Geometry'], mesh_boolean_difference.inputs[1])
                
            node_group.links.new(mesh_boolean_difference.outputs['Mesh'], transform_geometry.inputs['Geometry'])
            node_group.links.new(transform_geometry.outputs['Geometry'], mesh_to_volume.inputs['Mesh'])
            node_group.links.new(mesh_to_volume.outputs['Volume'], distribute_points.inputs['Volume'])
            node_group.links.new(distribute_points.outputs['Points'], points_to_vertices.inputs['Points'])
            node_group.links.new(points_to_vertices.outputs['Mesh'], group_output.inputs['Geometry'])
        bpy.data.objects.remove(domain_obj_copy, do_unlink=True)   
        geo_nodes_modifier = obj.modifiers.new(name=node_group_name, type='NODES')
        geo_nodes_modifier.node_group = node_group
        return node_group

    
    def create_geometry_node_system(self, obj, particle_diameter, domain_obj):
        self.delete_existing_geometry_node_group(obj)
        node_group_name = f"DF.GeometryNodes_Fluid_{obj.name}"
        if node_group_name in bpy.data.node_groups:
            node_group = bpy.data.node_groups[node_group_name]
        else:
            node_group = bpy.data.node_groups.new(name=node_group_name, type='GeometryNodeTree')
            
            group_input = node_group.nodes.new('NodeGroupInput')
            group_output = node_group.nodes.new('NodeGroupOutput')
            object_info_domain = node_group.nodes.new('GeometryNodeObjectInfo')
            transform_scale = node_group.nodes.new('GeometryNodeTransform')
            mesh_boolean_intersect = node_group.nodes.new('GeometryNodeMeshBoolean')
            mesh_boolean_difference = node_group.nodes.new('GeometryNodeMeshBoolean')
            join_geometry_obstacles = node_group.nodes.new('GeometryNodeJoinGeometry')
            transform_geometry = node_group.nodes.new('GeometryNodeTransform')
            mesh_to_volume = node_group.nodes.new('GeometryNodeMeshToVolume')
            distribute_points = node_group.nodes.new('GeometryNodeDistributePointsInVolume')
            points_to_vertices = node_group.nodes.new('GeometryNodePointsToVertices')
            value_node = node_group.nodes.new('ShaderNodeValue')
            
            group_input.location = (-1600, 0)
            object_info_domain.location = (-1400, 300)
            mesh_boolean_intersect.location = (-1000, 0)
            join_geometry_obstacles.location = (-1000, -300)
            mesh_boolean_difference.location = (-600, 0)
            transform_geometry.location = (-400, 0)
            transform_scale.location = (-800, 300)
            mesh_to_volume.location = (-200, 0)
            distribute_points.location = (0, 0)
            points_to_vertices.location = (200, 0)
            group_output.location = (400, 0)
            value_node.location = (-400, -200)
            
            grid_size = float(particle_diameter*2)
            object_info_domain.inputs['Object'].default_value = domain_obj
            object_info_domain.transform_space = 'RELATIVE'
            domain_obj_copy = domain_obj.copy()
            domain_obj_copy.data = domain_obj.data.copy()
            bpy.context.collection.objects.link(domain_obj_copy)
            bpy.ops.object.select_all(action='DESELECT')
            domain_obj_copy.select_set(True)
            bpy.ops.object.transform_apply(location=True, rotation=True, scale=True)
            bpy.ops.object.origin_set(type='ORIGIN_GEOMETRY', center='MEDIAN')
            dimensions = domain_obj_copy.dimensions
            shrink_factor_x = 1.0 - (grid_size / dimensions.x)
            shrink_factor_y = 1.0 - (grid_size / dimensions.y)
            shrink_factor_z = 1.0 - (grid_size / dimensions.z)
            
            mesh_boolean_intersect.operation = 'INTERSECT'
            mesh_boolean_difference.operation = 'DIFFERENCE'
            mesh_to_volume.resolution_mode = "VOXEL_SIZE"
            mesh_to_volume.inputs['Voxel Size'].default_value = 0.1
            distribute_points.mode = 'DENSITY_GRID'
            distribute_points.inputs['Spacing'].default_value[0] = particle_diameter/obj.scale.x
            distribute_points.inputs['Spacing'].default_value[1] = particle_diameter/obj.scale.y
            distribute_points.inputs['Spacing'].default_value[2] = particle_diameter/obj.scale.z
            distribute_points.inputs['Threshold'].default_value = 0.1
            # value_node.outputs[0].default_value = particle_diameter
            value_node.outputs[0].default_value = 0.1
            
            transform_scale.inputs['Scale'].default_value[0] = shrink_factor_x
            transform_scale.inputs['Scale'].default_value[1] = shrink_factor_y
            transform_scale.inputs['Scale'].default_value[2] = shrink_factor_z
            
            transform_geometry.inputs['Translation'].default_value = obj.location
            transform_geometry.inputs['Scale'].default_value = obj.scale
            transform_geometry.inputs['Rotation'].default_value = obj.rotation_euler
            
            node_group.interface.new_socket(name="Geometry", description="Geometry input", in_out='INPUT', socket_type='NodeSocketGeometry')
            node_group.interface.new_socket(name="Geometry", description="Geometry output", in_out='OUTPUT', socket_type='NodeSocketGeometry')
            
            node_group.links.new(group_input.outputs['Geometry'], mesh_boolean_intersect.inputs[1])
            node_group.links.new(object_info_domain.outputs['Geometry'],transform_scale.inputs['Geometry'])
            node_group.links.new(transform_scale.outputs['Geometry'], mesh_boolean_intersect.inputs[1])
            
            node_group.links.new(mesh_boolean_intersect.outputs['Mesh'], mesh_boolean_difference.inputs[0])
            node_group.links.new(join_geometry_obstacles.outputs['Geometry'], mesh_boolean_difference.inputs[1])
                
            node_group.links.new(mesh_boolean_difference.outputs['Mesh'], transform_geometry.inputs['Geometry'])
            node_group.links.new(transform_geometry.outputs['Geometry'], mesh_to_volume.inputs['Mesh'])
            node_group.links.new(mesh_to_volume.outputs['Volume'], distribute_points.inputs['Volume'])
            node_group.links.new(distribute_points.outputs['Points'], points_to_vertices.inputs['Points'])
            node_group.links.new(points_to_vertices.outputs['Mesh'], group_output.inputs['Geometry'])
        bpy.data.objects.remove(domain_obj_copy, do_unlink=True)   
        geo_nodes_modifier = obj.modifiers.new(name=node_group_name, type='NODES')
        geo_nodes_modifier.node_group = node_group
        return node_group

    def export_vertices_to_txt(self, obj, cache_folder_path,original_name):
        file_path = os.path.join(cache_folder_path, f"{original_name}_voxelized_points.txt")
        depsgraph = bpy.context.evaluated_depsgraph_get()
        eval_obj = obj.evaluated_get(depsgraph)
        mesh = eval_obj.to_mesh()
        with open(file_path, 'w') as f:
            for vertex in mesh.vertices:
                f.write(f"{vertex.co.x:.6f}, {vertex.co.y:.6f}, {vertex.co.z:.6f}\n")
        eval_obj.to_mesh_clear()
        self.report({'INFO'}, f"Exported vertices to {file_path}")
    def export_vertices_to_vtk(self, obj, cache_folder_path, original_name):
        file_path = os.path.join(cache_folder_path, f"{original_name}_voxelized_points.vtk")
        depsgraph = bpy.context.evaluated_depsgraph_get()
        eval_obj = obj.evaluated_get(depsgraph)
        mesh = eval_obj.to_mesh()
        points = []
        for vertex in mesh.vertices:
            points.append([vertex.co.x, vertex.co.y, vertex.co.z])
        points_np = np.array(points, dtype=np.float32)
        self.write_vtk_points_polydata_legacy_binary(file_path, points_np)
        eval_obj.to_mesh_clear()
        self.report({'INFO'}, f"Exported vertices to {file_path}")
    def export_initialization_data(self, cache_folder_path, grid_num, particle_radius, fluid_particle_num, rigid_particle_num, keyframed_particle_num, total_particle_num):
        initialization_data_file_path = os.path.join(cache_folder_path, 'Initialization_data.txt')
        with open(initialization_data_file_path, 'w') as f:
            if fluid_particle_num > 0:
                f.write(f"Fluid particles: {fluid_particle_num}\n")
            if rigid_particle_num > 0:
                f.write(f"Solid objects particles: {rigid_particle_num}\n")
            if keyframed_particle_num > 0:
                f.write(f"Keyframed objects particles: {keyframed_particle_num}\n")
            f.write(f"Particle radius: {particle_radius}\n")
            f.write(f"Total no. particles within Domain: {total_particle_num}\n")
    def execute(self, context):
        bpy.context.scene.frame_set(1)
        blender_file_path = bpy.data.filepath
        blender_file_dir = os.path.dirname(blender_file_path)
        blender_file_name = ".".join(os.path.splitext(os.path.basename(blender_file_path))[:-1])
        cache_folder_name = f"{blender_file_name}_cache"
        cache_folder_path = os.path.join(blender_file_dir, cache_folder_name)
        objects = bpy.data.objects
        domain_obj = None
        obstacles = []
        keyframed_obstacles = []
        domain_resolution = 20 #reserved value
        domain_objects = [obj for obj in bpy.data.objects if obj.doriflow.object_type == 'TYPE_DOMAIN']
        if len(domain_objects) > 1:
            self.report({'ERROR'}, "More than one domain object found. Only one is allowed.")
            return {'CANCELLED'}
        elif len(domain_objects) == 1:
            domain_obj = domain_objects[0]
        for obj in objects:
            if obj.doriflow.object_type == 'TYPE_DOMAIN':
                domain_resolution = obj.doriflow.domain.resolution
                rounded_dimensions = np.ceil(np.array(obj.dimensions)).astype(int)
                largest_domain_size = max(rounded_dimensions)
                grid_size, particle_diameter, grid_num, particle_radius = calculate_grid_properties(largest_domain_size, domain_resolution)
            elif obj.doriflow.object_type == 'TYPE_OBSTACLE':
                obstacles.append(obj)
            elif obj.doriflow.object_type == 'TYPE_KEYFRAMED_OBSTACLE':
                keyframed_obstacles.append(obj)
        if domain_obj is None:
            self.report({'ERROR'}, "No domain object found.")
            return {'CANCELLED'}
        fluid_particle_num = 0
        rigid_particle_num = 0
        keyframed_particle_num = 0
        total_particle_num = 0
                
        for obj in objects:
            if obj.doriflow.object_type in ['TYPE_OBSTACLE', 'TYPE_KEYFRAMED_OBSTACLE']:
                original_name = obj.name
                voxelized_name = f"DF.{original_name}_initial_voxelized_particles"
                # Check if the voxelized object already exists
                existing_obj = bpy.data.objects.get(voxelized_name)
                if existing_obj:
                    bpy.data.objects.remove(existing_obj, do_unlink=True)
                obj_copy = obj.copy()
                obj_copy.data = obj.data.copy()
                obj_copy.data.name = voxelized_name
                obj_copy.name = voxelized_name
                bpy.context.collection.objects.link(obj_copy)
                obj_copy.doriflow.object_type = "TYPE_NONE"
                obj_copy.animation_data_clear()
                bpy.ops.object.select_all(action='DESELECT')
                obj_copy.select_set(True)
                bpy.ops.object.transform_apply(location=True, rotation=True, scale=True)
                bpy.ops.object.origin_set(type='ORIGIN_GEOMETRY', center='MEDIAN')
                node_group = self.create_geometry_node_system(obj_copy, particle_diameter, domain_obj)
                self.export_vertices_to_txt(obj_copy, cache_folder_path,original_name)
                depsgraph = bpy.context.evaluated_depsgraph_get()
                eval_obj = obj_copy.evaluated_get(depsgraph)
                mesh = eval_obj.to_mesh()
                vertex_count = len(mesh.vertices)
                eval_obj.to_mesh_clear()
                if obj.doriflow.object_type == 'TYPE_OBSTACLE':
                    rigid_particle_num += vertex_count
                elif obj.doriflow.object_type == 'TYPE_KEYFRAMED_OBSTACLE':
                    keyframed_particle_num += vertex_count
                total_particle_num += vertex_count
                bpy.data.objects.remove(obj_copy, do_unlink=True)
                bpy.data.node_groups.remove(node_group, do_unlink=True)
                for block in bpy.data.meshes:
                    if block.users == 0:
                        bpy.data.meshes.remove(block)

                
            elif obj.doriflow.object_type =='TYPE_FLUID':
                original_name = obj.name
                voxelized_name = f"DF.{original_name}_initial_voxelized_particles"
                existing_obj = bpy.data.objects.get(voxelized_name)
                if existing_obj:
                    bpy.data.objects.remove(existing_obj, do_unlink=True)
                obj_copy = obj.copy()
                obj_copy.data = obj.data.copy()
                obj_copy.data.name = voxelized_name
                obj_copy.name = voxelized_name
                bpy.context.collection.objects.link(obj_copy)
                obj_copy.doriflow.object_type = "TYPE_NONE"
                obj_copy.animation_data_clear()
                bpy.ops.object.select_all(action='DESELECT')
                obj_copy.select_set(True)
                bpy.ops.object.transform_apply(location=True, rotation=True, scale=True)
                bpy.ops.object.origin_set(type='ORIGIN_GEOMETRY', center='MEDIAN')
                node_group = self.create_geometry_node_system_for_fluid(obj_copy, particle_diameter, obj, domain_obj,obstacles, keyframed_obstacles)
                self.export_vertices_to_txt(obj_copy, cache_folder_path,original_name)
                depsgraph = bpy.context.evaluated_depsgraph_get()
                eval_obj = obj_copy.evaluated_get(depsgraph)
                mesh = eval_obj.to_mesh()
                vertex_count = len(mesh.vertices)
                eval_obj.to_mesh_clear()
                if obj.doriflow.object_type == 'TYPE_FLUID':
                    fluid_particle_num += vertex_count
                total_particle_num += vertex_count
                bpy.data.objects.remove(obj_copy, do_unlink=True)
                bpy.data.node_groups.remove(node_group, do_unlink=True)
                for block in bpy.data.meshes:
                    if block.users == 0:
                        bpy.data.meshes.remove(block)

            
        self.export_initialization_data(
            cache_folder_path, grid_num, particle_radius, fluid_particle_num, 
            rigid_particle_num, keyframed_particle_num, total_particle_num)

        return {'FINISHED'}

class DORIFLOW_OT_PREVIEWVoxelizationGeometryNodes(bpy.types.Operator):
    bl_idname = "doriflow.preview_voxelization_geometry_nodes"
    bl_label = "Voxelization Geometry Nodes"
    bl_description = (
                    "Initialize the simulation with particles for the selected resolution. "
                    "Be cautious with extremely high resolutions as they may freeze the system. "
                    "Increase the resolution if there are not enough particles or if fluid leaks through solid walls. "
                    "Decrease the resolution if the system runs out of memory or becomes lagging.")
    
    def write_vtk_points_polydata_legacy_binary(self,
        file_path,
        points,
        vectors_dict=None,
        scalars_dict=None
    ):
        if vectors_dict is None:
            vectors_dict = {}
        if scalars_dict is None:
            scalars_dict = {}
        num_points = len(points)
        points_f32 = points.astype(np.float32)
        points_be = points_f32.byteswap().tobytes()
        connectivity = np.zeros((num_points, 2), dtype=np.int32)
        for i in range(num_points):
            connectivity[i, 0] = 1  # Number of points in the vertex
            connectivity[i, 1] = i  # Point index
        connectivity_be = connectivity.byteswap().tobytes()
        with open(file_path, "wb") as f:
            # Write Header
            header = (
                "# vtk DataFile Version 3.0\n"
                "Binary point cloud data\n"
                "BINARY\n"
                "DATASET POLYDATA\n"
            )
            f.write(header.encode("utf-8"))
            # Write POINTS
            f.write(f"POINTS {num_points} float\n".encode("utf-8"))
            f.write(points_be)
            # Write VERTICES
            f.write(f"VERTICES {num_points} {2 * num_points}\n".encode("utf-8"))
            f.write(connectivity_be)
            # Write POINT_DATA
            f.write(f"POINT_DATA {num_points}\n".encode("utf-8"))
            # --- Write Scalars ---
            for name, data in scalars_dict.items():
                if len(data) != num_points:
                    raise ValueError(f"Scalar array length mismatch for '{name}'")
                f.write(f"SCALARS {name} float 1\nLOOKUP_TABLE default\n".encode("utf-8"))
                data_f32 = data.astype(np.float32)
                data_be = data_f32.byteswap().tobytes()
                f.write(data_be)
            # --- Write Vectors ---
            for name, data in vectors_dict.items():
                if len(data) != num_points:
                    raise ValueError(f"Vector array length mismatch for '{name}'")
                f.write(f"VECTORS {name} float\n".encode("utf-8"))
                data_f32 = data.astype(np.float32)
                data_be = data_f32.byteswap().tobytes()
                f.write(data_be)
    def delete_existing_geometry_node_group(self, obj):
        node_group_names = [
            f"DF.GeometryNodes_{obj.name}",
            f"DF.GeometryNodes_Fluid_{obj.name}"]
        for node_group_name in node_group_names:
            if node_group_name in bpy.data.node_groups:
                bpy.data.node_groups.remove(bpy.data.node_groups[node_group_name], do_unlink=True)
    def create_geometry_node_system_for_fluid(self, obj, particle_diameter, fluid_obj, domain_obj,obstacles=None, keyframed_obstacles=None):
        self.delete_existing_geometry_node_group(obj)
        node_group_name = f"DF.GeometryNodes_Fluid_{obj.name}"
        if node_group_name in bpy.data.node_groups:
            node_group = bpy.data.node_groups[node_group_name]
        else:
            node_group = bpy.data.node_groups.new(name=node_group_name, type='GeometryNodeTree')
            
            group_input = node_group.nodes.new('NodeGroupInput')
            group_output = node_group.nodes.new('NodeGroupOutput')
            object_info_domain = node_group.nodes.new('GeometryNodeObjectInfo')
            transform_geometry = node_group.nodes.new('GeometryNodeTransform')
            mesh_to_volume = node_group.nodes.new('GeometryNodeMeshToVolume')
            distribute_points = node_group.nodes.new('GeometryNodeDistributePointsInVolume')
            points_to_vertices = node_group.nodes.new('GeometryNodePointsToVertices')
            value_node = node_group.nodes.new('ShaderNodeValue')
            
            object_info_obstacles = []
            if obstacles:
                for i, obstacle in enumerate(obstacles):
                    obj_info = node_group.nodes.new('GeometryNodeObjectInfo')
                    obj_info.inputs['Object'].default_value = obstacle
                    obj_info.transform_space = 'RELATIVE'
                    object_info_obstacles.append(obj_info)
            
            object_info_keyframed = []
            if keyframed_obstacles:
                for i, keyframed_obstacle in enumerate(keyframed_obstacles):
                    obj_info = node_group.nodes.new('GeometryNodeObjectInfo')
                    obj_info.inputs['Object'].default_value = keyframed_obstacle
                    obj_info.transform_space = 'RELATIVE'
                    object_info_keyframed.append(obj_info)
            
            group_input.location = (-1600, 0)
            object_info_domain.location = (-1400, 300)
            transform_geometry.location = (-400, 0)
            mesh_to_volume.location = (-200, 0)
            distribute_points.location = (0, 0)
            points_to_vertices.location = (200, 0)
            group_output.location = (400, 0)
            value_node.location = (-400, -200)
            
            object_info_domain.inputs['Object'].default_value = domain_obj
            object_info_domain.transform_space = 'RELATIVE'
            domain_obj_copy = domain_obj.copy()
            domain_obj_copy.data = domain_obj.data.copy()
            bpy.context.collection.objects.link(domain_obj_copy)
            bpy.ops.object.select_all(action='DESELECT')
            domain_obj_copy.select_set(True)
            bpy.ops.object.transform_apply(location=True, rotation=True, scale=True)
            bpy.ops.object.origin_set(type='ORIGIN_GEOMETRY', center='MEDIAN')
            dimensions = domain_obj_copy.dimensions
            
            mesh_to_volume.resolution_mode = "VOXEL_SIZE"
            mesh_to_volume.inputs['Voxel Size'].default_value = 0.1
            distribute_points.mode = 'DENSITY_GRID'
            distribute_points.inputs['Spacing'].default_value[0] = particle_diameter/obj.scale.x
            distribute_points.inputs['Spacing'].default_value[1] = particle_diameter/obj.scale.y
            distribute_points.inputs['Spacing'].default_value[2] = particle_diameter/obj.scale.z
            distribute_points.inputs['Threshold'].default_value = 0.1
            value_node.outputs[0].default_value = 0.1
            
            transform_geometry.inputs['Translation'].default_value = obj.location
            transform_geometry.inputs['Scale'].default_value = obj.scale
            transform_geometry.inputs['Rotation'].default_value = obj.rotation_euler
            
            node_group.interface.new_socket(name="Geometry", description="Geometry input", in_out='INPUT', socket_type='NodeSocketGeometry')
            node_group.interface.new_socket(name="Geometry", description="Geometry output", in_out='OUTPUT', socket_type='NodeSocketGeometry')
            
            node_group.links.new(group_input.outputs['Geometry'],transform_geometry.inputs['Geometry'])
            node_group.links.new(transform_geometry.outputs['Geometry'], mesh_to_volume.inputs['Mesh'])
            node_group.links.new(mesh_to_volume.outputs['Volume'], distribute_points.inputs['Volume'])
            node_group.links.new(distribute_points.outputs['Points'], group_output.inputs['Geometry'])
        bpy.data.objects.remove(domain_obj_copy, do_unlink=True)   
        geo_nodes_modifier = obj.modifiers.new(name=node_group_name, type='NODES')
        geo_nodes_modifier.node_group = node_group
        return node_group

    
    def create_geometry_node_system(self, obj, particle_diameter, domain_obj):
        self.delete_existing_geometry_node_group(obj)
        node_group_name = f"DF.GeometryNodes_Fluid_{obj.name}"
        if node_group_name in bpy.data.node_groups:
            node_group = bpy.data.node_groups[node_group_name]
        else:
            node_group = bpy.data.node_groups.new(name=node_group_name, type='GeometryNodeTree')
            
            group_input = node_group.nodes.new('NodeGroupInput')
            group_output = node_group.nodes.new('NodeGroupOutput')
            transform_geometry = node_group.nodes.new('GeometryNodeTransform')
            mesh_to_volume = node_group.nodes.new('GeometryNodeMeshToVolume')
            distribute_points = node_group.nodes.new('GeometryNodeDistributePointsInVolume')
            value_node = node_group.nodes.new('ShaderNodeValue')
            
            group_input.location = (-1600, 0)
            transform_geometry.location = (-400, 0)
            mesh_to_volume.location = (-200, 0)
            distribute_points.location = (0, 0)
            group_output.location = (400, 0)
            value_node.location = (-400, -200)
            domain_obj_copy = domain_obj.copy()
            domain_obj_copy.data = domain_obj.data.copy()
            bpy.context.collection.objects.link(domain_obj_copy)
            bpy.ops.object.select_all(action='DESELECT')
            domain_obj_copy.select_set(True)
            bpy.ops.object.transform_apply(location=True, rotation=True, scale=True)
            bpy.ops.object.origin_set(type='ORIGIN_GEOMETRY', center='MEDIAN')
            dimensions = domain_obj_copy.dimensions
            mesh_to_volume.resolution_mode = "VOXEL_SIZE"
            mesh_to_volume.inputs['Voxel Size'].default_value = 0.1
            distribute_points.mode = 'DENSITY_GRID'
            distribute_points.inputs['Spacing'].default_value[0] = particle_diameter/obj.scale.x
            distribute_points.inputs['Spacing'].default_value[1] = particle_diameter/obj.scale.y
            distribute_points.inputs['Spacing'].default_value[2] = particle_diameter/obj.scale.z
            distribute_points.inputs['Threshold'].default_value = 0.1
            value_node.outputs[0].default_value = 0.1
            
            
            transform_geometry.inputs['Translation'].default_value = obj.location
            transform_geometry.inputs['Scale'].default_value = obj.scale
            transform_geometry.inputs['Rotation'].default_value = obj.rotation_euler
            
            node_group.interface.new_socket(name="Geometry", description="Geometry input", in_out='INPUT', socket_type='NodeSocketGeometry')
            node_group.interface.new_socket(name="Geometry", description="Geometry output", in_out='OUTPUT', socket_type='NodeSocketGeometry')
            
            node_group.links.new(group_input.outputs['Geometry'], transform_geometry.inputs['Geometry'])
            node_group.links.new(transform_geometry.outputs['Geometry'], mesh_to_volume.inputs['Mesh'])
            node_group.links.new(mesh_to_volume.outputs['Volume'], distribute_points.inputs['Volume'])
            node_group.links.new(distribute_points.outputs['Points'], group_output.inputs['Geometry'])
        bpy.data.objects.remove(domain_obj_copy, do_unlink=True)   
        geo_nodes_modifier = obj.modifiers.new(name=node_group_name, type='NODES')
        geo_nodes_modifier.node_group = node_group
        return node_group

    def export_vertices_to_txt(self, obj, cache_folder_path,original_name):
        file_path = os.path.join(cache_folder_path, f"{original_name}_voxelized_points.txt")
        depsgraph = bpy.context.evaluated_depsgraph_get()
        eval_obj = obj.evaluated_get(depsgraph)
        mesh = eval_obj.to_mesh()
        with open(file_path, 'w') as f:
            for vertex in mesh.vertices:
                f.write(f"{vertex.co.x:.6f}, {vertex.co.y:.6f}, {vertex.co.z:.6f}\n")
        eval_obj.to_mesh_clear()
        self.report({'INFO'}, f"Exported vertices to {file_path}")
    
    def export_point_cloud_to_txt(self,obj, cache_folder_path,original_name):
        file_path = os.path.join(cache_folder_path, f"{original_name}_voxelized_points.txt")
        if not obj or obj.type != 'POINTCLOUD':
            print(f"Object {obj.name} is not a point cloud. Skipping export.")
            return
        try:
            point_cloud = obj.data
            with open(file_path, 'w') as file:
                for point in point_cloud.points:
                    coords = point.co
                    file.write(f"{coords.x} {coords.y} {coords.z}\n")
            print(f"Point cloud for {obj.name} exported to {file_path}")
        except Exception as e:
            print(f"Failed to export point cloud for {obj.name}: {e}")
    
    def export_initialization_data(self, cache_folder_path, grid_num, particle_radius, fluid_particle_num, rigid_particle_num, keyframed_particle_num, total_particle_num):
        initialization_data_file_path = os.path.join(cache_folder_path, 'Initialization_data.txt')
        with open(initialization_data_file_path, 'w') as f:
            if fluid_particle_num > 0:
                f.write(f"Fluid particles: {fluid_particle_num}\n")
            if rigid_particle_num > 0:
                f.write(f"Solid objects particles: {rigid_particle_num}\n")
            if keyframed_particle_num > 0:
                f.write(f"Keyframed objects particles: {keyframed_particle_num}\n")
            f.write(f"Particle radius: {particle_radius}\n")
            f.write(f"Total no. particles within Domain: {total_particle_num}\n")
    def execute(self, context):
        bpy.context.scene.frame_set(1)
        blender_file_path = bpy.data.filepath
        blender_file_dir = os.path.dirname(blender_file_path)
        blender_file_name = ".".join(os.path.splitext(os.path.basename(blender_file_path))[:-1])
        cache_folder_name = f"{blender_file_name}_cache"
        cache_folder_path = os.path.join(blender_file_dir, cache_folder_name)
        objects = bpy.data.objects
        domain_obj = None
        obstacles = []
        keyframed_obstacles = []
        domain_resolution = 20 #reserved value
        domain_objects = [obj for obj in bpy.data.objects if obj.doriflow.object_type == 'TYPE_DOMAIN']
        if len(domain_objects) > 1:
            self.report({'ERROR'}, "More than one domain object found. Only one is allowed.")
            return {'CANCELLED'}
        elif len(domain_objects) == 1:
            domain_obj = domain_objects[0]
        for obj in objects:
            if obj.doriflow.object_type == 'TYPE_DOMAIN':
                domain_resolution = obj.doriflow.domain.resolution
                rounded_dimensions = np.ceil(np.array(obj.dimensions)).astype(int)
                largest_domain_size = max(rounded_dimensions)
                grid_size, particle_diameter, _, _ = calculate_grid_properties(largest_domain_size, domain_resolution)
            elif obj.doriflow.object_type == 'TYPE_OBSTACLE':
                obstacles.append(obj)
            elif obj.doriflow.object_type == 'TYPE_KEYFRAMED_OBSTACLE':
                keyframed_obstacles.append(obj)
        if domain_obj is None:
            self.report({'ERROR'}, "No domain object found.")
            return {'CANCELLED'}
        fluid_particle_num = 0
        rigid_particle_num = 0
        keyframed_particle_num = 0
        total_particle_num = 0
                
        for obj in objects:
            if obj.doriflow.object_type in ['TYPE_OBSTACLE', 'TYPE_KEYFRAMED_OBSTACLE']:
                original_name = obj.name
                voxelized_name = f"DF.{original_name}_initial_voxelized_particles"
                existing_obj = bpy.data.objects.get(voxelized_name)
                if existing_obj:
                    bpy.data.objects.remove(existing_obj, do_unlink=True)
                obj_copy = obj.copy()
                obj_copy.data = obj.data.copy()
                obj_copy.name = voxelized_name
                bpy.context.collection.objects.link(obj_copy)
                obj_copy.doriflow.object_type = "TYPE_NONE"
                obj_copy.animation_data_clear()
                bpy.ops.object.select_all(action='DESELECT')
                obj_copy.select_set(True)
                bpy.ops.object.transform_apply(location=True, rotation=True, scale=True)
                node_group = self.create_geometry_node_system(obj_copy, particle_diameter, domain_obj)
                depsgraph = bpy.context.evaluated_depsgraph_get()
                eval_obj = obj_copy.evaluated_get(depsgraph)
                mesh = eval_obj.to_mesh()
                vertex_count = len(mesh.vertices)
                eval_obj.to_mesh_clear()
                if obj.doriflow.object_type == 'TYPE_OBSTACLE':
                    rigid_particle_num += vertex_count
                elif obj.doriflow.object_type == 'TYPE_KEYFRAMED_OBSTACLE':
                    keyframed_particle_num += vertex_count
                total_particle_num += vertex_count
                bpy.ops.object.select_all(action='DESELECT')
                obj.hide_set(True)
                
            elif obj.doriflow.object_type =='TYPE_FLUID':
                original_name = obj.name
                voxelized_name = f"DF.{original_name}_initial_voxelized_particles"

                existing_obj = bpy.data.objects.get(voxelized_name)
                if existing_obj:
                    bpy.data.objects.remove(existing_obj, do_unlink=True)
                obj_copy = obj.copy()
                obj_copy.data = obj.data.copy()
                obj_copy.data.name = voxelized_name
                obj_copy.name = voxelized_name
                bpy.context.collection.objects.link(obj_copy)
                obj_copy.doriflow.object_type = "TYPE_NONE"
                obj_copy.animation_data_clear()
                bpy.ops.object.select_all(action='DESELECT')
                obj_copy.select_set(True)
                bpy.ops.object.transform_apply(location=True, rotation=True, scale=True)
                node_group = self.create_geometry_node_system_for_fluid(obj_copy, particle_diameter, obj, domain_obj,obstacles, keyframed_obstacles)
                depsgraph = bpy.context.evaluated_depsgraph_get()
                eval_obj = obj_copy.evaluated_get(depsgraph)
                mesh = eval_obj.to_mesh()
                vertex_count = len(mesh.vertices)
                eval_obj.to_mesh_clear()
                if obj.doriflow.object_type == 'TYPE_FLUID':
                    fluid_particle_num += vertex_count
                total_particle_num += vertex_count
                bpy.ops.object.select_all(action='DESELECT')
                obj.select_set(True)
                obj.hide_set(True)
                for block in bpy.data.meshes:
                    if block.users == 0:
                        bpy.data.meshes.remove(block)
        return {'FINISHED'}

def register():
    bpy.utils.register_class(DORIFLOW_OT_VoxelizationGeometryNodes)
    bpy.utils.register_class(DORIFLOW_OT_PREVIEWVoxelizationGeometryNodes)

def unregister():
    bpy.utils.unregister_class(DORIFLOW_OT_VoxelizationGeometryNodes)
    bpy.utils.unregister_class(DORIFLOW_OT_PREVIEWVoxelizationGeometryNodes)

if __name__ == "__main__":
    register()

