# Doriflow Engine - Fluid Simulation for Blender 3D
# Copyright (C) 2024 Doriflow Team
# This software is licensed under the Creative Commons Attribution-NonCommercial 4.0 International License (CC BY-NC 4.0).

# You are free to:
# -Share: Copy and redistribute the material in any medium or format.
# -Adapt: Remix, transform, and build upon the material.

# UNDER THE FOLOWING TERMS:
# -Attribution: You must give appropriate credit, provide a link to the license, and indicate if changes were made.
# -Appropriate credit should include the following:
#   -The original author's name: Doriflow Team
#   -A link to the original source (if applicable).
#   -A link to the full license: https://creativecommons.org/licenses/by-nc/4.0/.
#   -A clear indication of any changes made, such as: "This material has been modified."
# NonCommercial: You may not use the material for commercial purposes.
# Disclaimer:
# -This simulation engine is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. In no event shall the authors be liable for any claim, damages, or other liability arising from the use of this software.

# For more details, refer to the full license text at:
# https://creativecommons.org/licenses/by-nc/4.0/.

#-----------------------------------------------------------------------------------------------------------------------#
import json
import bpy
from bpy.props import (EnumProperty, PointerProperty, StringProperty, BoolProperty)
from bpy.types import PropertyGroup
from . import domain_properties, fluid_properties, obstacle_properties, keyframed_obstacle_properties, inlet_properties, outlet_properties, grain_properties


class DoriflowObjectProperties(PropertyGroup):
    object_type: EnumProperty(
        name='Type',
        description='Type of participation in the Doriflow simulation',
        items=[
            ('TYPE_NONE', 'None', ''),
            ('TYPE_DOMAIN', 'Domain', 'Bounding box of this object represents the computational domain of the fluid simulation'),
            ('TYPE_FLUID', 'Fluid', 'Object represents a volume of fluid in the simulation'),
            ('TYPE_OBSTACLE', 'Obstacle', 'Object represents an obstacle in the simulation'),
            ('TYPE_KEYFRAMED_OBSTACLE', 'Keyframed Obstacle', 'Object represents a keyframed obstacle in the simulation'),
            ('TYPE_INLET', 'Inlet', 'Object represents an inlet in the simulation'),
            ('TYPE_OUTLET', 'Outlet', 'Object represents an outlet in the simulation'),
            ('TYPE_GRAIN', 'Grain', 'Object represents a grain in the simulation')
        ],
        default='TYPE_NONE'
    )
    domain: PointerProperty(
        name='Doriflow Domain Properties',
        type=domain_properties.DoriflowDomainProperties
    )
    fluid: PointerProperty(
        name='Doriflow Fluid Properties',
        type=fluid_properties.DoriflowFluidProperties
    )
    obstacle: PointerProperty(
        name='Doriflow Obstacle Properties',
        type=obstacle_properties.DoriflowObstacleProperties
    )
    keyframed_obstacle: PointerProperty(
        name='Doriflow Keyframed Obstacle Properties',
        type=keyframed_obstacle_properties.DoriflowKeyframedObstacleProperties
    )
    inlet: PointerProperty(
        name='Doriflow Inlet Properties',
        type=inlet_properties.DoriflowInletProperties
    )
    outlet: PointerProperty(
        name='Doriflow Outlet Properties',
        type=outlet_properties.DoriflowOutletProperties
    )
    grain: PointerProperty(
        name='Doriflow Grain Properties',
        type=grain_properties.DoriflowGrainProperties
    )
    obj_path: StringProperty(subtype="FILE_PATH")
    voxelized_points_path: StringProperty(subtype="FILE_PATH")
    is_active: BoolProperty(default=False)

    @classmethod
    def register(cls):
        bpy.types.Object.doriflow = PointerProperty(name='Doriflow Object Properties', type=cls)

    @classmethod
    def unregister(cls):
        del bpy.types.Object.doriflow

    def is_domain(self):
        return self.object_type == 'TYPE_DOMAIN'


def register():
    domain_properties.register()
    fluid_properties.register()
    obstacle_properties.register()
    keyframed_obstacle_properties.register()
    inlet_properties.register()
    outlet_properties.register()
    grain_properties.register()
    bpy.utils.register_class(DoriflowObjectProperties)


def unregister():
    bpy.utils.unregister_class(DoriflowObjectProperties)
    grain_properties.unregister()
    outlet_properties.unregister()
    inlet_properties.unregister()
    obstacle_properties.unregister()
    keyframed_obstacle_properties.unregister()
    fluid_properties.unregister()
    domain_properties.unregister()
