# Doriflow Engine - Fluid Simulation for Blender 3D
# Copyright (C) 2024 Doriflow Team
# This software is licensed under the Creative Commons Attribution-NonCommercial 4.0 International License (CC BY-NC 4.0).

# You are free to:
# -Share: Copy and redistribute the material in any medium or format.
# -Adapt: Remix, transform, and build upon the material.

# UNDER THE FOLOWING TERMS:
# -Attribution: You must give appropriate credit, provide a link to the license, and indicate if changes were made.
# -Appropriate credit should include the following:
#   -The original author's name: Doriflow Team
#   -A link to the original source (if applicable).
#   -A link to the full license: https://creativecommons.org/licenses/by-nc/4.0/.
#   -A clear indication of any changes made, such as: "This material has been modified."
# NonCommercial: You may not use the material for commercial purposes.
# Disclaimer:
# -This simulation engine is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. In no event shall the authors be liable for any claim, damages, or other liability arising from the use of this software.

# For more details, refer to the full license text at:
# https://creativecommons.org/licenses/by-nc/4.0/.

#-----------------------------------------------------------------------------------------------------------------------#
import json
import bpy
import os
from pathlib import Path
from bpy.props import EnumProperty, PointerProperty, StringProperty
from bpy.types import PropertyGroup, Operator


class PresetProperties(PropertyGroup):
    presets_scenes: EnumProperty(
        name="Presets",
        description="Select a simulation preset",
        items=lambda self, context: PresetProperties.get_preset_items(context),
    )
    presets_folder_path: StringProperty(
        name="Presets Folder Path",
        description="Path to the folder containing the presets",
        default="",
        subtype='DIR_PATH'
    )

    _cached_preset_items = None

    @classmethod
    def get_preset_items(cls, context):
        if cls._cached_preset_items is not None:
            return cls._cached_preset_items

        items = []
        try:
            current_file_path = os.path.realpath(__file__)
            current_directory = Path(current_file_path).parent
            default_presets_folder_path = current_directory.parent / "presets"
            if default_presets_folder_path.exists() and default_presets_folder_path.is_dir():
                default_presets_dirs = next(os.walk(default_presets_folder_path))[1]
                default_presets_dirs = sorted(default_presets_dirs, key=lambda d: int(''.join(filter(str.isdigit, d))))
                items.extend([(d, d.replace('_', ' ').title(), "") for d in default_presets_dirs])
            user_presets_folder_path = bpy.path.abspath(context.scene.preset_properties.presets_folder_path)
            if os.path.exists(user_presets_folder_path) and os.path.isdir(user_presets_folder_path):
                user_presets_dirs = next(os.walk(user_presets_folder_path))[1]
                user_presets_dirs = sorted(user_presets_dirs, key=lambda d: int(''.join(filter(str.isdigit, d))))
                items.extend([(d, d.replace('_', ' ').title(), "") for d in user_presets_dirs])
            items = list(dict.fromkeys(items))
        except Exception as e:
            items = []

        cls._cached_ppreset_items = items
        return items

def register():
    bpy.utils.register_class(PresetProperties)
    bpy.types.Scene.preset_properties = PointerProperty(type=PresetProperties)
    
def unregister():
    bpy.utils.unregister_class(PresetProperties)
    del bpy.types.Scene.preset_properties
