# Doriflow Engine - Fluid Simulation for Blender 3D
# Copyright (C) 2024 Doriflow Team
# This software is licensed under the Creative Commons Attribution-NonCommercial 4.0 International License (CC BY-NC 4.0).

# You are free to:
# -Share: Copy and redistribute the material in any medium or format.
# -Adapt: Remix, transform, and build upon the material.

# UNDER THE FOLOWING TERMS:
# -Attribution: You must give appropriate credit, provide a link to the license, and indicate if changes were made.
# -Appropriate credit should include the following:
#   -The original author's name: Doriflow Team
#   -A link to the original source (if applicable).
#   -A link to the full license: https://creativecommons.org/licenses/by-nc/4.0/.
#   -A clear indication of any changes made, such as: "This material has been modified."
# NonCommercial: You may not use the material for commercial purposes.
# Disclaimer:
# -This simulation engine is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. In no event shall the authors be liable for any claim, damages, or other liability arising from the use of this software.

# For more details, refer to the full license text at:
# https://creativecommons.org/licenses/by-nc/4.0/.

#-----------------------------------------------------------------------------------------------------------------------#
import json
import bpy
import os

def display_initialization_data():
    blender_file_path = bpy.data.filepath
    blender_file_dir = os.path.dirname(blender_file_path)
    blender_file_name = os.path.splitext(os.path.basename(blender_file_path))[0]
    cache_folder_name = f"{blender_file_name}_cache"
    cache_folder_path = os.path.join(blender_file_dir, cache_folder_name)
    initialization_data_path = os.path.join(cache_folder_path, "Initialization_data_from_compute.txt")
    data_lines = []
    if os.path.exists(initialization_data_path):
        try:
            with open(initialization_data_path, 'r') as f:
                data = f.read()
                data_lines = [line.strip() for line in data.split('\n') if line.strip()]
        except Exception as e:
            print(f"Error reading initialization data: {e}")
            data_lines = ["Error displaying initialization data."]
    else:
        data_lines = ["No initialization data available."]
    return data_lines

def draw_compute_fluid_operator(context, layout):
    global_props = context.scene.doriflow
    if global_props is None:
        return

    compute_fluid_props = global_props.compute_fluid
    if not compute_fluid_props.is_simulation_running:
        operator_text = "Compute Fluid and Solid Motions"
    elif not compute_fluid_props.is_compute_fluid_initialized:
        operator_text = "Loading initial state..."
    elif compute_fluid_props.is_compute_fluid_paused:
        frame_str = str(compute_fluid_props.num_compute_fluid_frames)
        operator_text = "Resume Simulation from frame " + frame_str 
    elif compute_fluid_props.is_compute_fluid_stopped:
        if compute_fluid_props.is_safe_to_exit:
            safety_str = "Safe to quit Blender"
        else:
            safety_str = "Do NOT quit Blender"
        operator_text = "Cancelling... " + safety_str
    else:
        num_frames = compute_fluid_props.num_total_frames
        frame_progress_string = str(compute_fluid_props.num_compute_fluid_frames) + " / " + str(num_frames)
        pct_string = str(round((compute_fluid_props.num_compute_fluid_frames / num_frames) * 100, 1)) + "%"
        frames_string = pct_string + "  (" + frame_progress_string + ")"
        operator_text = "Computing in progress...   " + frames_string
    column = layout.column(align=True)
    if compute_fluid_props.is_compute_fluid_paused:
        split = column.split(factor=0.75, align=True)
        column_left = split.column(align=True)
        column_left.operator("doriflow.resume_compute_fluid_simulation", text=operator_text)

        column_right = split.column(align=True)
        column_right.alert = True
        column_right.operator("doriflow.stop_compute_fluid_simulation", text="Stop")
    else:
        column.operator("doriflow.compute_fluid_simulation", text=operator_text, icon='MOD_FLUIDSIM')
        if compute_fluid_props.is_simulation_running:
            column.operator("doriflow.pause_compute_fluid_simulation", text="Pause")
        
class DORIFLOW_PT_GlobalSettingsPanel(bpy.types.Panel):
    bl_space_type = "PROPERTIES"
    bl_region_type = "WINDOW"
    bl_context = "physics"
    bl_category = "Doriflow"
    bl_label = "[Doriflow] -2- Main Simulation"
    bl_order = 10
    bl_options = {'DEFAULT_CLOSED'}

    @classmethod
    def poll(cls, context):
        return context.object is not None and context.object.doriflow.object_type == 'TYPE_DOMAIN' and context.object.doriflow.is_active

    def draw(self, context):
        layout = self.layout
        box = layout.box()
        draw_compute_fluid_operator(context, box) 
        box.prop(context.scene.doriflow, "show_init_data", text="Initialization Data")
        if context.scene.doriflow.show_init_data:
            initialization_data = display_initialization_data()
            data_column = box.column(align=True)
            for line in initialization_data:
                data_column.label(text=line)
        row = box.row()
        col = row.column()
        col.scale_x = 0.5 
        col.label(text="Import:", icon ='IMPORT')  
        col = row.column()
        col.scale_x = 0.5
        solid_motion_button = col.row()
        solid_motion_button.operator("doriflow.import_motion_data", text="Solid Motion")
def register():
    bpy.utils.register_class(DORIFLOW_PT_GlobalSettingsPanel)

def unregister():
    bpy.utils.unregister_class(DORIFLOW_PT_GlobalSettingsPanel)
